"""
Admin API for Intent Management
واجهة إدارة النوايا والقواعد
"""

from fastapi import APIRouter, HTTPException, Body
from typing import Dict, List, Optional
from pydantic import BaseModel
from app.intents.unified_classifier import unified_classifier
from app.intents.rules_loader import intent_rules_loader
from app.services.metrics_tracker import metrics_tracker
from app.utils.logger import app_logger as logger

router = APIRouter(prefix="/api/admin/intents", tags=["Admin - Intents"])


# ========================================
# Request/Response Models
# ========================================

class TestClassificationRequest(BaseModel):
    """طلب اختبار التصنيف"""
    message: str
    context: Optional[Dict] = {}


class TestClassificationResponse(BaseModel):
    """رد اختبار التصنيف"""
    message: str
    intent: Optional[str]
    confidence: Optional[float]
    reason: Optional[str]
    classification_method: str
    response_time_ms: float


class ReloadRulesResponse(BaseModel):
    """رد إعادة تحميل القواعد"""
    success: bool
    message: str
    stats: Dict


# ========================================
# Endpoints
# ========================================

@router.get("/stats")
async def get_intent_stats():
    """
    الحصول على إحصائيات النظام
    
    Returns:
        - إحصائيات القواعد المحملة
        - إحصائيات الكلمات المفتاحية
        - إحصائيات الأمثلة
    """
    try:
        stats = unified_classifier.get_stats()
        
        return {
            "success": True,
            "stats": stats
        }
        
    except Exception as e:
        logger.error(f"❌ Error getting intent stats: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/rules")
async def get_all_rules():
    """
    الحصول على جميع القواعد
    
    Returns:
        قائمة بجميع القواعد المحملة
    """
    try:
        rules = intent_rules_loader.get_all_rules()
        
        rules_data = []
        for rule in rules:
            rules_data.append({
                "name": rule.name,
                "intent_type": rule.intent_type.value,
                "priority": rule.priority,
                "confidence": rule.confidence,
                "description": rule.description,
                "patterns": rule.patterns,
                "keywords": rule.keywords,
                "exclusions": rule.exclusions,
                "context_required": rule.context_required,
                "examples": rule.examples
            })
        
        return {
            "success": True,
            "total_rules": len(rules_data),
            "rules": rules_data
        }
        
    except Exception as e:
        logger.error(f"❌ Error getting rules: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/rules/{intent_name}")
async def get_rule_by_intent(intent_name: str):
    """
    الحصول على قاعدة نية معينة
    
    Args:
        intent_name: اسم النية (مثل: GREETING, ORDER_FOOD)
    """
    try:
        from app.intents.intent_types import IntentType
        
        # Convert name to IntentType
        try:
            intent_type = IntentType[intent_name.upper()]
        except KeyError:
            raise HTTPException(status_code=404, detail=f"Intent '{intent_name}' not found")
        
        rule = intent_rules_loader.get_rule(intent_type)
        
        if not rule:
            raise HTTPException(status_code=404, detail=f"Rule for intent '{intent_name}' not found")
        
        return {
            "success": True,
            "rule": {
                "name": rule.name,
                "intent_type": rule.intent_type.value,
                "priority": rule.priority,
                "confidence": rule.confidence,
                "description": rule.description,
                "patterns": rule.patterns,
                "keywords": rule.keywords,
                "exclusions": rule.exclusions,
                "context_required": rule.context_required,
                "examples": rule.examples
            }
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"❌ Error getting rule: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.post("/test")
async def test_classification(request: TestClassificationRequest):
    """
    اختبار تصنيف رسالة
    
    Args:
        message: الرسالة المراد تصنيفها
        context: السياق (اختياري)
    
    Returns:
        النية المكتشفة ودرجة الثقة
    """
    try:
        import time
        
        start_time = time.time()
        
        # Test with unified classifier
        result = unified_classifier.classify(
            message=request.message,
            context=request.context
        )
        
        end_time = time.time()
        response_time_ms = (end_time - start_time) * 1000
        
        if result:
            intent_type, confidence, reason = result
            
            return TestClassificationResponse(
                message=request.message,
                intent=intent_type.value,
                confidence=confidence,
                reason=reason,
                classification_method="rule_based",
                response_time_ms=response_time_ms
            )
        else:
            return TestClassificationResponse(
                message=request.message,
                intent=None,
                confidence=None,
                reason="No rule match found",
                classification_method="none",
                response_time_ms=response_time_ms
            )
        
    except Exception as e:
        logger.error(f"❌ Error testing classification: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.post("/reload")
async def reload_rules():
    """
    إعادة تحميل القواعد من الملفات
    
    يستخدم لتحديث القواعد بدون إعادة تشغيل السيرفر
    """
    try:
        logger.info("🔄 Reloading intent rules...")
        
        success = unified_classifier.reload_rules()
        
        if success:
            stats = unified_classifier.get_stats()
            
            return ReloadRulesResponse(
                success=True,
                message="Rules reloaded successfully",
                stats=stats
            )
        else:
            return ReloadRulesResponse(
                success=False,
                message="Failed to reload rules",
                stats={}
            )
        
    except Exception as e:
        logger.error(f"❌ Error reloading rules: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/keywords/{category}")
async def get_keywords_by_category(category: str, lang: str = "ar"):
    """
    الحصول على كلمات مفتاحية لفئة معينة
    
    Args:
        category: الفئة (مثل: food_categories.appetizers)
        lang: اللغة (ar أو en)
    
    Examples:
        - /keywords/food_categories.appetizers?lang=ar
        - /keywords/dialects.gulf.keywords.want?lang=ar
    """
    try:
        keywords = intent_rules_loader.get_keywords_for_category(category, lang)
        
        return {
            "success": True,
            "category": category,
            "lang": lang,
            "keywords": keywords,
            "count": len(keywords)
        }
        
    except Exception as e:
        logger.error(f"❌ Error getting keywords: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/examples/{intent_name}")
async def get_examples_by_intent(intent_name: str):
    """
    الحصول على أمثلة لنية معينة
    
    Args:
        intent_name: اسم النية (مثل: GREETING, ORDER_FOOD)
    """
    try:
        examples = intent_rules_loader.get_examples_for_intent(intent_name.upper())
        
        return {
            "success": True,
            "intent": intent_name.upper(),
            "examples": examples,
            "count": len(examples)
        }
        
    except Exception as e:
        logger.error(f"❌ Error getting examples: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/metrics/summary")
async def get_metrics_summary():
    """
    الحصول على ملخص مقاييس الأداء
    
    Returns:
        - نسبة نجاح القواعد vs AI
        - متوسط وقت الاستجابة
        - معدل النجاح
        - النوايا الأكثر استخداماً
    """
    try:
        summary = metrics_tracker.get_summary()
        
        return {
            "success": True,
            "metrics": summary
        }
        
    except Exception as e:
        logger.error(f"❌ Error getting metrics summary: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/metrics/detailed")
async def get_detailed_metrics():
    """
    الحصول على مقاييس مفصلة
    
    Returns:
        - إحصائيات مفصلة
        - توزيع ساعي
        - أداء حديث
    """
    try:
        detailed = metrics_tracker.get_detailed_stats()
        
        return {
            "success": True,
            "metrics": detailed
        }
        
    except Exception as e:
        logger.error(f"❌ Error getting detailed metrics: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.get("/metrics/comparison")
async def get_performance_comparison():
    """
    مقارنة أداء القواعد vs AI
    
    Returns:
        - مقارنة وقت الاستجابة
        - مقارنة معدل النجاح
        - مقارنة درجة الثقة
    """
    try:
        comparison = metrics_tracker.get_performance_comparison()
        
        return {
            "success": True,
            "comparison": comparison
        }
        
    except Exception as e:
        logger.error(f"❌ Error getting performance comparison: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))


@router.post("/metrics/save")
async def save_daily_metrics():
    """
    حفظ المقاييس اليومية
    """
    try:
        metrics_tracker.save_daily_metrics()
        
        return {
            "success": True,
            "message": "Daily metrics saved successfully"
        }
        
    except Exception as e:
        logger.error(f"❌ Error saving metrics: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))

