"""
🌟 نظام الردود الذكية للتحيات (Context-Aware Greeting Replies)

هذا الملف يحتوي على أكثر من 100 رد متنوع للتحيات
مصنفة حسب نوع التحية لتكون الردود طبيعية ومناسبة للسياق

التصنيفات:
- 🕌 ردود السلام (Islamic Greetings)
- 👋 ردود المرحبا (Welcome Greetings)
- 🤗 ردود هلا (Gulf Greetings)
- ☀️ ردود الصباح (Morning Greetings)
- 🌙 ردود المساء (Evening Greetings)
- 🌟 ردود عامة (General Greetings)
- 🌍 ردود إنجليزية (English Greetings)
"""

import random
from typing import List, Dict

# 🕌 ردود السلام (Islamic Greetings)
SALAM_REPLIES = [
    "وعليكم السلام ورحمة الله وبركاته 🌿\nأهلاً وسهلاً بك في مطعم الذوق العماني 🍽️",
    "وعليكم السلام ورحمة الله 🌟\nنورت مطعمنا! كيف أقدر أخدمك اليوم؟",
    "عليكم السلام والرحمة 👋\nتشرفنا بزيارتك! وش تحب تطلب؟",
    "وعليكم السلام 🌿\nحياك الله في مطعم الذوق العماني، تفضل بالطلب",
    "السلام عليكم ورحمة الله 🌟\nأهلاً بك! جاهزين نخدمك بأطيب الأطباق",
    "وعليكم السلام والرحمة والبركات 🌿\nمنور المطعم! وش نجهز لك اليوم؟",
    "عليكم السلام 👋\nيا هلا! تفضل وش تشتهي من قائمتنا؟",
    "وعليكم السلام ورحمة الله 🌟\nحياك! عندنا أطيب الأكلات العمانية",
]

# 👋 ردود المرحبا (Welcome Greetings)
MARHABA_REPLIES = [
    "مرحبتين 👋\nنورت مطعم الذوق العماني! وش تحب تطلب اليوم؟",
    "يا مرحبا وأهلاً وسهلاً 🌸\nتفضل، قائمتنا جاهزة لك",
    "مراحب 🤗\nحياك الله! عندنا أطيب الأطباق العمانية",
    "أهلين وسهلين 👋\nمنور المطعم! وش نجهز لك؟",
    "يا أهلا بالطيب 🌟\nتشرفنا! تفضل بالطلب",
    "مرحبا بك في بيتك الثاني 🏠\nوش تشتهي من قائمتنا؟",
    "أهلاً وسهلاً ومرحباً 🌸\nجاهزين نخدمك بأطيب الأكل",
    "يا مرحبتين بالغالي 🤗\nتفضل، كل قائمتنا تحت أمرك",
    "أهلا بك 👋\nنورتنا! وش نحضر لك من أطباقنا الشهية؟",
    "مرحبا فيك 🌟\nحياك الله! عندنا كل اللي تشتهيه",
]

# 🤗 ردود هلا (Gulf Greetings)
HALA_REPLIES = [
    "هلا وغلا 🤗\nنورت مطعم الذوق العماني! وش تحب تطلب؟",
    "هلا والله بالطيب 🌟\nتفضل، قائمتنا جاهزة",
    "يا هلا بالغالي 👋\nمنور! وش نجهز لك اليوم؟",
    "هلا فيك والله 🤗\nحياك! عندنا أطيب الأكلات",
    "هلا والمرحبا 🌟\nتشرفنا بزيارتك! تفضل بالطلب",
    "هلا بك يا الغالي 👋\nوش تشتهي من قائمتنا؟",
    "يا هلا ومرحبا 🤗\nجاهزين نخدمك بأطيب الأطباق",
    "هلا والله نورت 🌟\nتفضل، كل القائمة تحت أمرك",
    "هلا بالطيب 👋\nحياك الله! وش نحضر لك؟",
    "يا هلا بمن نور المكان 🌟\nتفضل بالطلب، عندنا كل اللي تحبه",
]

# ☀️ ردود الصباح (Morning Greetings)
SABAH_REPLIES = [
    "صباح النور والسرور ☀️\nنورت مطعمنا! وش تحب تفطر اليوم؟",
    "صباح الخير والبركة 🌤️\nتفضل، عندنا أطيب فطور عماني",
    "صباح الورد والياسمين 🌸\nحياك! وش نجهز لك من قائمة الفطور؟",
    "صباح الفل ☀️\nمنور! تفضل بالطلب",
    "صباحك سعيد ومبارك 🌤️\nعندنا أطيب الأكلات لبداية يومك",
    "صباح الخيرات والبركات ☀️\nتشرفنا! وش تشتهي للفطور؟",
    "صباح مشرق ☀️\nحياك الله! جاهزين نخدمك بأطيب الأطباق",
    "صباحك ورد وفل 🌸\nتفضل، قائمتنا جاهزة لك",
    "صباح السعادة والأمل ☀️\nنورت! وش نحضر لك اليوم؟",
    "صباح النور يا الغالي 🌤️\nعندنا كل اللي يبدأ يومك بطاقة",
]

# 🌙 ردود المساء (Evening Greetings)
MASA_REPLIES = [
    "مساء النور والسرور 🌙\nنورت مطعمنا! وش تحب تتعشى اليوم؟",
    "مساء الخير والبركة 🌆\nتفضل، عندنا أطيب عشاء عماني",
    "مساء الورد والياسمين 🌹\nحياك! وش نجهز لك من قائمة العشاء؟",
    "مساء الفل 🌙\nمنور! تفضل بالطلب",
    "مساءك سعيد ومبارك 🌆\nعندنا أطيب الأكلات لعشاء مميز",
    "مساء الخيرات 🌙\nتشرفنا! وش تشتهي للعشاء؟",
    "مساء جميل 🌆\nحياك الله! جاهزين نخدمك بأطيب الأطباق",
    "مساءك ورد وفل 🌹\nتفضل، قائمتنا جاهزة لك",
    "مساء السعادة 🌙\nنورت! وش نحضر لك اليوم؟",
    "مساء النور يا الغالي 🌆\nعندنا كل اللي تشتهيه لعشاء لذيذ",
]

# 🌟 ردود عامة (General Greetings)
GENERAL_REPLIES = [
    "حياك الله 🌟\nنورت مطعم الذوق العماني! وش تحب تطلب؟",
    "الله يحييك 👋\nتفضل، قائمتنا جاهزة لك",
    "نورت المكان 🌟\nحياك! عندنا أطيب الأطباق العمانية",
    "تشرفنا بزيارتك 👋\nوش نجهز لك اليوم؟",
    "منور يا الغالي 🌟\nتفضل بالطلب، عندنا كل اللي تحبه",
    "حياك وبياك 👋\nوش تشتهي من قائمتنا؟",
    "يا مرحبا بالطيب 🌟\nجاهزين نخدمك بأطيب الأكل",
    "نورت بوجودك 👋\nتفضل، كل القائمة تحت أمرك",
    "أهلاً بك 🌟\nحياك الله! وش نحضر لك؟",
    "يا هلا بمن نور 👋\nتفضل بالطلب، عندنا أطيب الأطباق",
]

# 🌍 ردود إنجليزية (English Greetings)
ENGLISH_REPLIES = [
    "Hello! 👋\nWelcome to Omani Taste Restaurant! What would you like to order?",
    "Hi there! 🌟\nGreat to see you! Check out our menu",
    "Hey! 👋\nWelcome! We have the best Omani dishes for you",
    "Good to see you! 🌟\nWhat can we prepare for you today?",
    "Welcome! 👋\nOur menu is ready for you",
    "Hi! 🌟\nGlad you're here! What would you like to try?",
    "Hello and welcome! 👋\nWe're ready to serve you the finest dishes",
    "Hey there! 🌟\nWhat can we get you today?",
    "Greetings! 👋\nWelcome to our restaurant! Feel free to order",
    "Hi! Welcome! 🌟\nLet us know what you'd like to eat",
]

# 📋 قاموس الردود الرئيسي
GREETING_REPLIES: Dict[str, List[str]] = {
    "salam": SALAM_REPLIES,
    "marhaba": MARHABA_REPLIES,
    "hala": HALA_REPLIES,
    "sabah": SABAH_REPLIES,
    "masa": MASA_REPLIES,
    "general": GENERAL_REPLIES,
    "english": ENGLISH_REPLIES,
}


def get_greeting_reply(greeting_type: str = "general") -> str:
    """
    الحصول على رد مناسب للتحية
    
    Args:
        greeting_type: نوع التحية (salam, marhaba, hala, sabah, masa, general, english)
    
    Returns:
        str: رد مناسب للتحية
    """
    replies = GREETING_REPLIES.get(greeting_type, GENERAL_REPLIES)
    return random.choice(replies)


def detect_greeting_type(message: str) -> str:
    """
    كشف نوع التحية من الرسالة
    
    Args:
        message: رسالة المستخدم
    
    Returns:
        str: نوع التحية (salam, marhaba, hala, sabah, masa, general, english)
    """
    message_lower = message.lower().strip()
    
    # 🕌 تحيات السلام
    if any(word in message_lower for word in ["السلام", "سلام", "سلامات", "سلامو"]):
        return "salam"
    
    # ☀️ تحيات الصباح
    elif any(word in message_lower for word in ["صباح", "صباحك", "صباحو"]):
        return "sabah"
    
    # 🌙 تحيات المساء
    elif any(word in message_lower for word in ["مساء", "مساءك", "مساءو"]):
        return "masa"
    
    # 🤗 تحيات هلا (أولوية أعلى من مرحبا)
    elif any(word in message_lower for word in ["هلا", "يا هلا", "هلا والله", "هلا وغلا", "هلا فيك"]):
        return "hala"
    
    # 👋 تحيات المرحبا
    elif any(word in message_lower for word in ["مرحبا", "مرحبتين", "مراحب", "أهلا", "أهلين", "اهلا", "اهلين"]):
        return "marhaba"
    
    # 🌍 تحيات إنجليزية
    elif any(word in message_lower for word in ["hello", "hi", "hey", "good morning", "good evening", "good afternoon", "greetings", "howdy"]):
        return "english"
    
    # 🌟 تحيات عامة (افتراضي)
    else:
        return "general"


def get_smart_greeting_reply(message: str) -> str:
    """
    الحصول على رد ذكي للتحية بناءً على نوعها
    
    Args:
        message: رسالة المستخدم
    
    Returns:
        str: رد مناسب للتحية
    """
    greeting_type = detect_greeting_type(message)
    return get_greeting_reply(greeting_type)


# 📊 إحصائيات التحيات
GREETING_STATS = {
    "salam": len(SALAM_REPLIES),
    "marhaba": len(MARHABA_REPLIES),
    "hala": len(HALA_REPLIES),
    "sabah": len(SABAH_REPLIES),
    "masa": len(MASA_REPLIES),
    "general": len(GENERAL_REPLIES),
    "english": len(ENGLISH_REPLIES),
}

TOTAL_GREETINGS = sum(GREETING_STATS.values())

if __name__ == "__main__":
    print("🌟 نظام الردود الذكية للتحيات")
    print("=" * 50)
    print(f"\n📊 إجمالي الردود: {TOTAL_GREETINGS} رد\n")
    
    for greeting_type, count in GREETING_STATS.items():
        print(f"  • {greeting_type}: {count} رد")
    
    print("\n" + "=" * 50)
    print("\n🧪 اختبار الردود:\n")
    
    test_messages = [
        "السلام عليكم",
        "مرحبا",
        "هلا",
        "صباح الخير",
        "مساء الخير",
        "حياك",
        "hello"
    ]
    
    for msg in test_messages:
        greeting_type = detect_greeting_type(msg)
        reply = get_smart_greeting_reply(msg)
        print(f"👤 {msg}")
        print(f"🤖 [{greeting_type}] {reply}\n")

