"""
Bilingual Response Templates
قوالب الردود ثنائية اللغة (عربي/إنجليزي)
"""

from typing import Dict, Optional, List
from enum import Enum
import random


class ResponseType(Enum):
    """أنواع الردود"""
    GREETING = "greeting"
    GOODBYE = "goodbye"
    THANKS_REPLY = "thanks_reply"
    ITEM_ADDED = "item_added"
    ITEM_NOT_FOUND = "item_not_found"
    ASK_QUANTITY = "ask_quantity"
    ASK_ADD_MORE = "ask_add_more"
    SHOW_CART = "show_cart"
    CART_EMPTY = "cart_empty"
    ASK_DELIVERY_ADDRESS = "ask_delivery_address"
    ASK_PAYMENT_METHOD = "ask_payment_method"
    ORDER_CONFIRMED = "order_confirmed"
    ORDER_CANCELLED = "order_cancelled"
    HELP = "help"
    CLARIFICATION = "clarification"
    ERROR = "error"


class BilingualResponseTemplates:
    """
    قوالب الردود ثنائية اللغة
    
    - ردود فورية بدون AI
    - دعم كامل للعربية والإنجليزية
    - تنويع في الردود (لا يكرر نفس الرد)
    """
    
    # القوالب الأساسية
    TEMPLATES = {
        ResponseType.GREETING: {
            'ar': [
                "مرحباً! 👋 أهلاً بك في مطعم {restaurant_name}",
                "أهلاً وسهلاً! 🌟 كيف يمكنني مساعدتك اليوم؟",
                "يا هلا! 😊 تفضل، كيف أقدر أخدمك؟",
                "السلام عليكم! 🙏 نورت المطعم"
            ],
            'en': [
                "Hello! 👋 Welcome to {restaurant_name}",
                "Hi there! 🌟 How can I help you today?",
                "Welcome! 😊 What can I get for you?",
                "Greetings! 🙏 Nice to have you here"
            ]
        },
        
        ResponseType.GOODBYE: {
            'ar': [
                "مع السلامة! 👋 نتمنى نشوفك قريب",
                "باي! 😊 شكراً لزيارتك",
                "الله يسلمك! 🌟 بانتظارك دائماً",
                "تصبح على خير! 🌙 (إذا كان مساءً)"
            ],
            'en': [
                "Goodbye! 👋 Hope to see you soon",
                "Bye! 😊 Thanks for visiting",
                "Take care! 🌟 See you next time",
                "Have a great day! ☀️"
            ]
        },
        
        ResponseType.THANKS_REPLY: {
            'ar': [
                "العفو! 😊",
                "على الرحب والسعة! 🌟",
                "تسلم! ❤️",
                "حياك الله! 🙏"
            ],
            'en': [
                "You're welcome! 😊",
                "My pleasure! 🌟",
                "Anytime! ❤️",
                "Glad to help! 🙏"
            ]
        },
        
        ResponseType.ITEM_ADDED: {
            'ar': [
                "تمام! ✅ تم إضافة {quantity} × {item_name} ({price} ر.ع)\n\n💰 المجموع: {total} ر.ع\n\n🤔 هل تريد إضافة صنف آخر؟",
                "ممتاز! ✨ أضفت {quantity} × {item_name} ({price} ر.ع)\n\n💵 الإجمالي: {total} ر.ع\n\n📝 تبغى تضيف شي ثاني؟"
            ],
            'en': [
                "Great! ✅ Added {quantity} × {item_name} ({price} OMR)\n\n💰 Total: {total} OMR\n\n🤔 Would you like to add another item?",
                "Perfect! ✨ {quantity} × {item_name} added ({price} OMR)\n\n💵 Total: {total} OMR\n\n📝 Anything else?"
            ]
        },
        
        ResponseType.ITEM_NOT_FOUND: {
            'ar': [
                "عذراً، ما حصلت '{item_name}' في المنيو 😕\n\n💡 جرب:\n• اكتب اسم الصنف بطريقة ثانية\n• اكتب 'المنيو' لعرض جميع الأصناف",
                "آسف، '{item_name}' مو موجود عندنا 🤷\n\nتبغى تشوف المنيو الكامل؟"
            ],
            'en': [
                "Sorry, couldn't find '{item_name}' in our menu 😕\n\n💡 Try:\n• Type the item name differently\n• Type 'menu' to see all items",
                "Oops, '{item_name}' is not available 🤷\n\nWould you like to see the full menu?"
            ]
        },
        
        ResponseType.ASK_QUANTITY: {
            'ar': [
                "كم واحد تبغى من {item_name}؟ 🔢",
                "كم العدد؟ (مثال: 2) 📝"
            ],
            'en': [
                "How many {item_name} would you like? 🔢",
                "Quantity? (e.g., 2) 📝"
            ]
        },
        
        ResponseType.ASK_ADD_MORE: {
            'ar': [
                "🤔 هل تريد إضافة صنف آخر؟",
                "تبغى تضيف شي ثاني؟ 🍽️"
            ],
            'en': [
                "🤔 Would you like to add another item?",
                "Anything else? 🍽️"
            ]
        },
        
        ResponseType.SHOW_CART: {
            'ar': [
                "🛒 **طلبك الحالي:**\n\n{items_list}\n\n💰 **المجموع:** {total} ر.ع\n\n---\n\n🤔 هل تريد:\n1️⃣ إضافة المزيد\n2️⃣ إكمال الطلب\n3️⃣ إلغاء الطلب"
            ],
            'en': [
                "🛒 **Your Cart:**\n\n{items_list}\n\n💰 **Total:** {total} OMR\n\n---\n\n🤔 Would you like to:\n1️⃣ Add more\n2️⃣ Complete order\n3️⃣ Cancel order"
            ]
        },
        
        ResponseType.CART_EMPTY: {
            'ar': [
                "السلة فاضية! 🛒\n\nتبغى تطلب شي؟ اكتب اسم الصنف أو 'المنيو' 📝",
                "ما عندك طلبات حالياً 🤷\n\nشو تبغى تطلب؟"
            ],
            'en': [
                "Your cart is empty! 🛒\n\nWhat would you like to order? Type item name or 'menu' 📝",
                "No items in cart yet 🤷\n\nWhat can I get you?"
            ]
        },
        
        ResponseType.ASK_DELIVERY_ADDRESS: {
            'ar': [
                "ممتاز! 🎉\n\n📍 أرسل موقعك الآن\n(اضغط 📎 ← Location)",
                "تمام! ✅\n\n📍 أرسل موقعك الآن\n(اضغط 📎 ← Location)"
            ],
            'en': [
                "Great! 🎉\n\n📍 Send your location now\n(Tap 📎 ← Location)",
                "Perfect! ✅\n\n📍 Send your location now\n(Tap 📎 ← Location)"
            ]
        },
        
        ResponseType.ASK_PAYMENT_METHOD: {
            'ar': [
                "تمام! 💳\n\nكيف تبغى تدفع؟\n\n1️⃣ كاش عند الاستلام 💵\n2️⃣ بطاقة ائتمان 💳\n3️⃣ تحويل بنكي 🏦",
                "اختر طريقة الدفع:\n\n• كاش 💵\n• بطاقة 💳\n• تحويل 🏦"
            ],
            'en': [
                "Great! 💳\n\nHow would you like to pay?\n\n1️⃣ Cash on delivery 💵\n2️⃣ Credit card 💳\n3️⃣ Bank transfer 🏦",
                "Choose payment method:\n\n• Cash 💵\n• Card 💳\n• Transfer 🏦"
            ]
        },
        
        ResponseType.ORDER_CONFIRMED: {
            'ar': [
                "🎉 **تم تأكيد طلبك!**\n\n📋 رقم الطلب: {order_number}\n💰 المجموع: {total} ر.ع\n📍 العنوان: {address}\n💳 الدفع: {payment_method}\n\n⏱️ سيصلك خلال 30-45 دقيقة\n\n📱 يمكنك تتبع طلبك بإرسال رقم الطلب\n\nشكراً لك! 🙏"
            ],
            'en': [
                "🎉 **Order Confirmed!**\n\n📋 Order #: {order_number}\n💰 Total: {total} OMR\n📍 Address: {address}\n💳 Payment: {payment_method}\n\n⏱️ Delivery in 30-45 minutes\n\n📱 Track your order by sending the order number\n\nThank you! 🙏"
            ]
        },
        
        ResponseType.ORDER_CANCELLED: {
            'ar': [
                "تم إلغاء الطلب ❌\n\nلا مشكلة! إذا بغيت تطلب مرة ثانية، أنا هنا 😊"
            ],
            'en': [
                "Order cancelled ❌\n\nNo problem! If you want to order again, I'm here 😊"
            ]
        },
        
        ResponseType.HELP: {
            'ar': [
                "💬 كيف أساعدك؟\n\n🍽️ اكتب اسم الصنف أو \"المنيو\"\n🛒 اكتب \"السلة\" لعرض طلبك\n📦 اكتب رقم الطلب للتتبع"
            ],
            'en': [
                "💬 How can I help?\n\n🍽️ Type item name or \"menu\"\n🛒 Type \"cart\" to view order\n📦 Type order number to track"
            ]
        },
        
        ResponseType.CLARIFICATION: {
            'ar': [
                "عذراً، ما فهمت 😕\n\nممكن توضح أكثر؟ أو اكتب 'مساعدة' للمساعدة 💡",
                "آسف، مو واضح 🤔\n\nجرب تكتب بطريقة ثانية؟"
            ],
            'en': [
                "Sorry, I didn't understand 😕\n\nCan you clarify? Or type 'help' for assistance 💡",
                "Hmm, not clear 🤔\n\nCould you rephrase?"
            ]
        },
        
        ResponseType.ERROR: {
            'ar': [
                "عذراً، حدث خطأ 😔\n\nجرب مرة ثانية، أو تواصل معنا: {contact}"
            ],
            'en': [
                "Sorry, an error occurred 😔\n\nPlease try again, or contact us: {contact}"
            ]
        }
    }
    
    @staticmethod
    def get(
        response_type: ResponseType,
        lang: str = 'ar',
        **kwargs
    ) -> str:
        """
        الحصول على رد جاهز
        
        Args:
            response_type: نوع الرد
            lang: اللغة ('ar' أو 'en')
            **kwargs: متغيرات للتعويض في القالب
        
        Returns:
            الرد الجاهز
        """
        # الحصول على القوالب
        templates = BilingualResponseTemplates.TEMPLATES.get(response_type, {})
        lang_templates = templates.get(lang, templates.get('ar', []))
        
        if not lang_templates:
            return "..."
        
        # اختيار قالب عشوائي (للتنويع)
        template = random.choice(lang_templates)
        
        # تعويض المتغيرات
        try:
            return template.format(**kwargs)
        except KeyError as e:
            # متغير ناقص
            return template
    
    @staticmethod
    def format_items_list(items: List[Dict], lang: str = 'ar') -> str:
        """تنسيق قائمة الأصناف"""
        if not items:
            return ""
        
        formatted = []
        for i, item in enumerate(items, 1):
            quantity = item.get('quantity', 1)
            name = item.get('name_ar' if lang == 'ar' else 'name_en', 'Unknown')
            price = item.get('price_omr', 0)
            total_price = quantity * price
            
            formatted.append(
                f"{i}. {quantity} × {name} = {total_price:.3f} ر.ع"
                if lang == 'ar' else
                f"{i}. {quantity} × {name} = {total_price:.3f} OMR"
            )
        
        return "\n".join(formatted)
    
    @staticmethod
    def get_greeting(lang: str = 'ar', restaurant_name: str = "مطعمنا") -> str:
        """رد التحية"""
        return BilingualResponseTemplates.get(
            ResponseType.GREETING,
            lang=lang,
            restaurant_name=restaurant_name
        )
    
    @staticmethod
    def get_item_added(
        item_name: str,
        quantity: int,
        price: float,
        total: float,
        lang: str = 'ar'
    ) -> str:
        """رد إضافة صنف"""
        return BilingualResponseTemplates.get(
            ResponseType.ITEM_ADDED,
            lang=lang,
            item_name=item_name,
            quantity=quantity,
            price=f"{price:.3f}",
            total=f"{total:.3f}"
        )
    
    @staticmethod
    def get_cart_summary(
        items: List[Dict],
        total: float,
        lang: str = 'ar'
    ) -> str:
        """عرض السلة"""
        items_list = BilingualResponseTemplates.format_items_list(items, lang)
        
        return BilingualResponseTemplates.get(
            ResponseType.SHOW_CART,
            lang=lang,
            items_list=items_list,
            total=f"{total:.3f}"
        )
    
    @staticmethod
    def get_order_confirmed(
        order_number: str,
        total: float,
        address: str,
        payment_method: str,
        lang: str = 'ar'
    ) -> str:
        """تأكيد الطلب"""
        return BilingualResponseTemplates.get(
            ResponseType.ORDER_CONFIRMED,
            lang=lang,
            order_number=order_number,
            total=f"{total:.3f}",
            address=address,
            payment_method=payment_method
        )


# Singleton instance
response_templates = BilingualResponseTemplates()

