"""
AI Response Models using Pydantic for Structured Outputs
These models enforce strict schemas that the AI must follow
"""
from pydantic import BaseModel, Field
from typing import Literal, List, Optional
from enum import Enum


# ============================================
# Enums for strict type checking
# ============================================

class IntentType(str, Enum):
    """All possible intents"""
    SMALL_TALK = "Small Talk"
    NEW_ORDER_ITEMS = "New Order - Items"
    VIEW_CART = "View Cart"  # NEW: Show current cart
    ASK_INFORMATION = "Ask Information"
    NEW_ORDER_ADDRESS = "New Order - Address"
    NEW_ORDER_PAYMENT = "New Order - Payment"
    CONFIRM_ORDER = "Confirm Order"
    CANCEL_ORDER = "Cancel Order"
    MODIFY_ORDER = "Modify Order"
    TRACK_ORDER = "Track Order"
    CONTINUE_ORDER = "Continue Order"
    REPEAT_ORDER = "Repeat Order"
    MENU_TEXT = "Menu Text"
    BUSINESS_HOURS = "Business Hours"
    LOCATION = "Location"
    OFFERS = "Offers"
    CONTACT_REQUEST = "Contact Request"


class PaymentMethod(str, Enum):
    """Payment methods"""
    CASH = "cash"
    CARD = "card"


# ============================================
# Order Management Responses
# ============================================

class OrderItem(BaseModel):
    """Single order item"""
    code: str = Field(description="Menu item code (e.g., MAIN_HAREES_MEAT)")
    name: str = Field(description="Menu item name in Arabic")
    quantity: int = Field(ge=1, le=99, description="Quantity (1-99)")
    price: float = Field(gt=0, description="Unit price in OMR")


class AddItemResponse(BaseModel):
    """Response when adding item(s) to order"""
    intent: Literal[IntentType.NEW_ORDER_ITEMS]
    reply: str = Field(description="Reply message in customer's language")
    action: Literal["update_session"]
    items: List[OrderItem] = Field(min_items=1, description="Items to add")
    subtotal: float = Field(ge=0, description="New subtotal in OMR")


class AskInformationResponse(BaseModel):
    """Response when asking for clarification"""
    intent: Literal[IntentType.ASK_INFORMATION]
    reply: str = Field(description="Question with numbered options in customer's language")
    options: List[str] = Field(min_items=2, description="List of options presented to customer")
    action: None = Field(default=None, description="Must be None for Ask Information")
    data: None = Field(default=None, description="Must be None for Ask Information")


class SmallTalkResponse(BaseModel):
    """Response for greetings and casual conversation"""
    intent: Literal[IntentType.SMALL_TALK]
    reply: str = Field(description="Friendly reply in customer's language")
    action: None = Field(default=None)
    data: None = Field(default=None)


class ViewCartResponse(BaseModel):
    """Response when customer asks to view their cart"""
    intent: Literal[IntentType.VIEW_CART]
    reply: str = Field(description="Show cart contents with items, quantities, prices, and total in customer's language")
    action: None = Field(default=None)
    data: None = Field(default=None)


# ============================================
# Address & Payment Responses
# ============================================

class AddressResponse(BaseModel):
    """Response when collecting delivery address"""
    intent: Literal[IntentType.NEW_ORDER_ADDRESS]
    reply: str = Field(description="Confirmation message in customer's language")
    action: Literal["update_session"]
    address: str = Field(description="Customer's delivery address")


class PaymentResponse(BaseModel):
    """Response when selecting payment method"""
    intent: Literal[IntentType.NEW_ORDER_PAYMENT]
    reply: str = Field(description="Confirmation message in customer's language")
    action: Literal["update_session"]
    payment_method: PaymentMethod = Field(description="Selected payment method")


# ============================================
# Order Actions Responses
# ============================================

class ConfirmOrderResponse(BaseModel):
    """Response when confirming order"""
    intent: Literal[IntentType.CONFIRM_ORDER]
    reply: str = Field(description="Order confirmation with summary in customer's language")
    action: Literal["confirm_order"]
    data: None = Field(default=None)


class CancelOrderResponse(BaseModel):
    """Response when canceling order"""
    intent: Literal[IntentType.CANCEL_ORDER]
    reply: str = Field(description="Cancellation confirmation in customer's language")
    action: Literal["cancel_order"]
    order_id: Optional[str] = Field(default=None, description="Order ID if canceling existing order")


class ModifyOrderResponse(BaseModel):
    """Response when modifying order"""
    intent: Literal[IntentType.MODIFY_ORDER]
    reply: str = Field(description="Modification confirmation in customer's language")
    action: Literal["update_session"]
    modification: str = Field(description="Description of modification")


class TrackOrderResponse(BaseModel):
    """Response when tracking order"""
    intent: Literal[IntentType.TRACK_ORDER]
    reply: str = Field(description="Order status information in customer's language")
    action: None = Field(default=None)
    order_id: str = Field(description="Order ID to track")


class ContinueOrderResponse(BaseModel):
    """Response when customer wants to continue/proceed"""
    intent: Literal[IntentType.CONTINUE_ORDER]
    reply: str = Field(description="Confirmation to proceed in customer's language")
    action: Literal["update_session"]
    next_stage: Literal["address", "payment", "confirmation"] = Field(
        description="Next stage to move to"
    )


class RepeatOrderResponse(BaseModel):
    """Response when customer wants to repeat last order"""
    intent: Literal[IntentType.REPEAT_ORDER]
    reply: str = Field(description="Confirmation with order details in customer's language")
    action: Literal["repeat_order"]
    data: None = Field(default=None)


# ============================================
# Information Responses
# ============================================

class MenuResponse(BaseModel):
    """Response when customer asks for menu"""
    intent: Literal[IntentType.MENU_TEXT]
    reply: str = Field(description="Menu information in customer's language")
    action: None = Field(default=None)
    data: None = Field(default=None)


class BusinessHoursResponse(BaseModel):
    """Response when customer asks about business hours"""
    intent: Literal[IntentType.BUSINESS_HOURS]
    reply: str = Field(description="Business hours information in customer's language")
    action: None = Field(default=None)
    data: None = Field(default=None)


class LocationResponse(BaseModel):
    """Response when customer asks about location"""
    intent: Literal[IntentType.LOCATION]
    reply: str = Field(description="Location information in customer's language")
    action: None = Field(default=None)
    data: None = Field(default=None)


class OffersResponse(BaseModel):
    """Response when customer asks about offers"""
    intent: Literal[IntentType.OFFERS]
    reply: str = Field(description="Offers information in customer's language")
    action: None = Field(default=None)
    data: None = Field(default=None)


class ContactRequestResponse(BaseModel):
    """Response when customer wants to contact support"""
    intent: Literal[IntentType.CONTACT_REQUEST]
    reply: str = Field(description="Contact information in customer's language")
    action: None = Field(default=None)
    data: None = Field(default=None)


# ============================================
# Union Type for All Responses
# ============================================

from typing import Union

AIResponse = Union[
    AddItemResponse,
    AskInformationResponse,
    SmallTalkResponse,
    AddressResponse,
    PaymentResponse,
    ConfirmOrderResponse,
    CancelOrderResponse,
    ModifyOrderResponse,
    TrackOrderResponse,
    ContinueOrderResponse,
    RepeatOrderResponse,
    MenuResponse,
    BusinessHoursResponse,
    LocationResponse,
    OffersResponse,
    ContactRequestResponse
]


# ============================================
# Helper function to get response model by stage
# ============================================

def get_response_model_for_stage(stage: str, has_items: bool = False):
    """
    Get the appropriate response model based on current stage
    
    Args:
        stage: Current order stage (items, address, payment, confirmation)
        has_items: Whether order has items (for determining if can proceed)
        
    Returns:
        Union type of possible response models for this stage
    """
    if stage == 'items' or not stage:
        # Items stage: can add items, ask for info, or do small talk
        return Union[AddItemResponse, AskInformationResponse, SmallTalkResponse,
                     ViewCartResponse, MenuResponse, ContinueOrderResponse, RepeatOrderResponse,
                     BusinessHoursResponse, LocationResponse, OffersResponse,
                     ContactRequestResponse, CancelOrderResponse]
    
    elif stage == 'address':
        return Union[AddressResponse, SmallTalkResponse, ModifyOrderResponse, CancelOrderResponse]
    
    elif stage == 'payment':
        return Union[PaymentResponse, SmallTalkResponse, ModifyOrderResponse, CancelOrderResponse]
    
    elif stage == 'confirmation':
        return Union[ConfirmOrderResponse, SmallTalkResponse, ModifyOrderResponse, CancelOrderResponse]
    
    else:
        # Default: allow most common responses
        return Union[SmallTalkResponse, TrackOrderResponse, RepeatOrderResponse,
                     MenuResponse, BusinessHoursResponse, LocationResponse,
                     OffersResponse, ContactRequestResponse]

# ============================================
# Unified Response Model (for OpenAI Structured Outputs)
# ============================================

class AIResponse(BaseModel):
    """
    Unified response model that supports all intent types
    OpenAI Structured Outputs doesn't support Union types,
    so we use a single flexible model with optional fields
    """
    intent: str = Field(description="Intent type (e.g., 'Small Talk', 'New Order - Items')")
    reply: str = Field(description="Reply message in customer's language")
    action: Optional[str] = Field(default=None, description="Action to perform (e.g., 'update_session', 'confirm_order')")

    # Optional fields for different response types
    items: Optional[List[OrderItem]] = Field(default=None, description="Items to add (for AddItemResponse)")
    subtotal: Optional[float] = Field(default=None, description="Subtotal in OMR (for AddItemResponse)")
    options: Optional[List[str]] = Field(default=None, description="Options list (for AskInformationResponse)")
    address: Optional[str] = Field(default=None, description="Delivery address (for AddressResponse)")
    payment_method: Optional[str] = Field(default=None, description="Payment method: 'cash' or 'card' (for PaymentResponse)")
    next_stage: Optional[str] = Field(default=None, description="Next stage for ContinueOrderResponse (e.g., 'address', 'payment')")
    order_number: Optional[str] = Field(default=None, description="Order number for tracking (for TrackOrderResponse)")


