"""
Main FastAPI Application
Restaurant Chatbot Server with Dynamic Prompt System
"""
from fastapi import FastAPI, Request
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import JSONResponse
from contextlib import asynccontextmanager
import time

from app.config import settings
from app.utils.logger import app_logger as logger
from app.routes import webhook, menu, credits


# Lifespan context manager for startup/shutdown events
@asynccontextmanager
async def lifespan(app: FastAPI):
    """Handle startup and shutdown events"""
    # Startup
    logger.info("=" * 60)
    logger.info("🚀 Starting Restaurant Chatbot Server")
    logger.info(f"Environment: {settings.ENVIRONMENT}")
    logger.info(f"Debug Mode: {settings.DEBUG}")
    logger.info(f"Server: {settings.SERVER_HOST}:{settings.SERVER_PORT}")
    logger.info(f"Evolution API: {settings.EVOLUTION_API_URL}")
    logger.info(f"OpenAI Model: {settings.OPENAI_MODEL}")
    logger.info("=" * 60)

    # Initialize FAISS menu search
    try:
        from app.services.faiss_menu_search import faiss_menu_search
        from app.services.local_menu_loader import load_local_menu

        if settings.USE_LOCAL_MENU:
            logger.info("🔄 Initializing FAISS menu search from local file...")
            menu_items = load_local_menu()
        else:
            logger.info("🔄 Initializing FAISS menu search from cache...")
            import json
            import os

            # Load menu items from cache file
            cache_path = os.path.join(
                os.path.dirname(__file__),
                '../data/cache/menu_cache.json'
            )

            with open(cache_path, 'r', encoding='utf-8') as f:
                cache_data = json.load(f)
                menu_items = cache_data.get('items', [])

        logger.info(f"📦 Loaded {len(menu_items)} items")

        # Build FAISS index
        await faiss_menu_search.load_menu(menu_items)

        stats = faiss_menu_search.get_stats()
        logger.info(f"✅ FAISS initialized: {stats['items_count']} items, {stats['memory_mb']:.1f} MB")

    except Exception as e:
        logger.error(f"❌ Failed to initialize FAISS: {e}")
        logger.warning("⚠️ Server will continue without FAISS (fallback to old search)")

    # Start reminder service background task
    try:
        from app.services.reminder_service import reminder_service
        import asyncio

        logger.info("🔄 Starting reminder service...")
        # Start the background task
        asyncio.create_task(reminder_service.start_background_task())
        logger.info("✅ Reminder service started")

    except Exception as e:
        logger.error(f"❌ Failed to start reminder service: {e}")
        logger.warning("⚠️ Server will continue without reminder service")

    yield

    # Shutdown
    logger.info("=" * 60)
    logger.info("🛑 Shutting down Restaurant Chatbot Server")
    logger.info("=" * 60)


# Create FastAPI app
app = FastAPI(
    title="Restaurant Chatbot Server",
    description="Standalone server for restaurant chatbot with dynamic prompt system",
    version="1.0.0",
    lifespan=lifespan
)


# CORS Middleware
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],  # Configure appropriately for production
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)


# Request logging middleware
@app.middleware("http")
async def log_requests(request: Request, call_next):
    """Log all incoming requests"""
    start_time = time.time()
    
    # Log request
    logger.info(f"→ {request.method} {request.url.path}")
    
    # Process request
    response = await call_next(request)
    
    # Calculate duration
    duration = time.time() - start_time
    
    # Log response
    logger.info(f"← {request.method} {request.url.path} - {response.status_code} ({duration:.3f}s)")
    
    return response


# Exception handler
@app.exception_handler(Exception)
async def global_exception_handler(request: Request, exc: Exception):
    """Handle all unhandled exceptions"""
    logger.error(f"Unhandled exception: {exc}", exc_info=True)
    
    return JSONResponse(
        status_code=500,
        content={
            "error": "Internal server error",
            "message": str(exc) if settings.DEBUG else "An error occurred"
        }
    )


# Include routers
app.include_router(webhook.router, tags=["Webhook"])
app.include_router(menu.router, tags=["Menu"])
app.include_router(credits.router, tags=["Credits"])

# Admin routers
from app.routes import admin_intents
app.include_router(admin_intents.router, tags=["Admin - Intents"])


# Root endpoint
@app.get("/")
async def root():
    """Root endpoint"""
    return {
        "service": "Restaurant Chatbot Server",
        "version": "1.0.0",
        "status": "running",
        "environment": settings.ENVIRONMENT,
        "features": {
            "dynamic_prompts": True,
            "evolution_api": True,
            "openai_integration": True,
            "supabase_backend": True
        }
    }


# Health check endpoint
@app.get("/health")
async def health_check():
    """
    Comprehensive health check endpoint
    Validates system configuration and connectivity
    """
    from supabase import create_client
    from datetime import datetime

    health_status = {
        "status": "healthy",
        "timestamp": datetime.now().isoformat(),
        "checks": {}
    }

    try:
        # Check 1: Server-side prompts validation
        from app.prompts import SYSTEM_ROLE, ORDER_MANAGEMENT_PROMPT, SMALL_TALK_PROMPT, RESPONSE_FORMAT_PROMPT

        health_status["checks"]["prompts"] = {
            "status": "ok",
            "system_role_loaded": len(SYSTEM_ROLE) > 0,
            "order_management_loaded": len(ORDER_MANAGEMENT_PROMPT) > 0,
            "small_talk_loaded": len(SMALL_TALK_PROMPT) > 0,
            "response_format_loaded": len(RESPONSE_FORMAT_PROMPT) > 0
        }

        # Check 2: Database connectivity
        supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)

        # Test database connection by fetching menu items count
        result = supabase.table('menu_items').select('code', count='exact').limit(1).execute()

        health_status["checks"]["database"] = {
            "status": "ok",
            "connected": True,
            "menu_items_accessible": True
        }

    except Exception as e:
        import traceback
        health_status["status"] = "unhealthy"
        health_status["checks"]["system"] = {
            "status": "error",
            "error": str(e),
            "traceback": traceback.format_exc()
        }

    return health_status


# Status endpoint with detailed information
@app.get("/status")
async def status():
    """Detailed status endpoint"""
    from supabase import create_client
    
    status_info = {
        "server": {
            "status": "running",
            "environment": settings.ENVIRONMENT,
            "debug": settings.DEBUG
        },
        "services": {
            "supabase": "unknown",
            "openai": "configured",
            "evolution_api": "configured"
        }
    }
    
    # Test Supabase connection
    try:
        supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)
        response = supabase.table('prompt_sections').select('count', count='exact').limit(1).execute()
        status_info["services"]["supabase"] = "connected"
    except Exception as e:
        status_info["services"]["supabase"] = f"error: {str(e)}"
        logger.error(f"Supabase connection test failed: {e}")
    
    return status_info


# Configuration endpoint (for debugging - remove in production)
@app.get("/config")
async def get_config():
    """Get current configuration (debug only)"""
    if not settings.DEBUG:
        return {"error": "Not available in production"}
    
    return {
        "server": {
            "host": settings.SERVER_HOST,
            "port": settings.SERVER_PORT,
            "environment": settings.ENVIRONMENT
        },
        "openai": {
            "model": settings.OPENAI_MODEL,
            "temperature": settings.OPENAI_TEMPERATURE,
            "max_tokens": settings.OPENAI_MAX_TOKENS
        },
        "evolution_api": {
            "url": settings.EVOLUTION_API_URL,
            "instance": settings.EVOLUTION_INSTANCE_NAME
        },
        "business": {
            "name": settings.RESTAURANT_NAME,
            "hours": f"{settings.BUSINESS_HOURS_START} - {settings.BUSINESS_HOURS_END}",
            "delivery_fee": settings.DELIVERY_FEE,
            "min_order": settings.MIN_ORDER_AMOUNT
        }
    }


if __name__ == "__main__":
    import uvicorn
    
    uvicorn.run(
        "app.main:app",
        host=settings.SERVER_HOST,
        port=settings.SERVER_PORT,
        reload=settings.DEBUG,
        log_level=settings.LOG_LEVEL.lower()
    )

