"""
Unified Intent Classifier
نظام تصنيف موحد يدمج جميع الطبقات
"""

from typing import Optional, Dict, List, Tuple
from app.intents.intent_types import IntentType
from app.intents.rules_loader import intent_rules_loader, IntentRule
from app.utils.logger import app_logger as logger
import re


class UnifiedIntentClassifier:
    """
    نظام تصنيف موحد للنوايا
    
    يدمج:
    1. Intent Guard (طبقة الحماية)
    2. Rule-Based Classifier (التصنيف بالقواعد)
    3. Context-Aware Classification (التصنيف الواعي بالسياق)
    
    التدفق:
    1. Pre-classification guard (كلمات مفتاحية عالية الأولوية)
    2. Pattern matching (أنماط regex)
    3. Keyword matching (كلمات مفتاحية)
    4. Context-based classification (السياق)
    5. Special patterns (أنماط خاصة)
    """
    
    def __init__(self):
        self.rules_loader = intent_rules_loader
        self.menu_keywords = set()  # Will be updated from database
        self._is_loaded = False
    
    def initialize(self) -> bool:
        """تهيئة النظام وتحميل القواعد"""
        if self._is_loaded:
            return True
        
        try:
            logger.info("🎯 Initializing Unified Intent Classifier...")
            
            # Load rules from files
            success = self.rules_loader.load_all()
            
            if not success:
                logger.error("❌ Failed to load intent rules")
                return False
            
            self._is_loaded = True
            
            # Log stats
            stats = self.rules_loader.get_stats()
            logger.info(
                f"✅ Unified Classifier initialized: "
                f"{stats['total_rules']} rules, "
                f"{stats['intents_covered']} intents"
            )
            
            return True
            
        except Exception as e:
            logger.error(f"❌ Error initializing classifier: {str(e)}")
            return False
    
    async def update_menu_keywords(self, menu_items: List[Dict]):
        """تحديث كلمات المنيو من قاعدة البيانات"""
        self.menu_keywords.clear()
        
        for item in menu_items:
            # Add Arabic name
            name_ar = item.get('name_ar', '').lower()
            if name_ar:
                self.menu_keywords.add(name_ar)
                # Add individual words
                for word in name_ar.split():
                    if len(word) > 2:
                        self.menu_keywords.add(word)
            
            # Add English name
            name_en = item.get('name_en', '').lower()
            if name_en:
                self.menu_keywords.add(name_en)
                for word in name_en.split():
                    if len(word) > 2:
                        self.menu_keywords.add(word)
        
        logger.info(f"✅ Updated menu keywords: {len(self.menu_keywords)} keywords")
    
    def classify(
        self,
        message: str,
        context: Dict
    ) -> Optional[Tuple[IntentType, float, str]]:
        """
        تصنيف النية
        
        Args:
            message: رسالة المستخدم
            context: السياق الحالي
        
        Returns:
            (IntentType, confidence, reason) أو None إذا لم يجد
        """
        if not self._is_loaded:
            logger.warning("⚠️ Classifier not initialized, initializing now...")
            self.initialize()
        
        message_lower = message.lower().strip()
        
        # Layer 0: Pre-classification guard (highest priority keywords)
        guard_result = self._pre_classify_guard(message_lower, context)
        if guard_result:
            intent_type, confidence, reason = guard_result
            logger.info(f"🛡️ Pre-guard: {intent_type.value} ({confidence:.2f}) - {reason}")
            return guard_result
        
        # Layer 1: Pattern matching (regex patterns)
        pattern_result = self._pattern_matching(message_lower, context)
        if pattern_result:
            intent_type, confidence, reason = pattern_result
            logger.info(f"✅ Pattern match: {intent_type.value} ({confidence:.2f}) - {reason}")
            return pattern_result
        
        # Layer 2: Special patterns (numbers, order IDs, etc.)
        special_result = self._special_patterns(message_lower, context)
        if special_result:
            intent_type, confidence, reason = special_result
            logger.info(f"🔢 Special pattern: {intent_type.value} ({confidence:.2f}) - {reason}")
            return special_result
        
        # Layer 3: Context-based classification
        context_result = self._context_based_classification(message_lower, context)
        if context_result:
            intent_type, confidence, reason = context_result
            logger.info(f"🎯 Context-based: {intent_type.value} ({confidence:.2f}) - {reason}")
            return context_result
        
        # Layer 4: Menu item detection
        menu_result = self._menu_item_detection(message_lower)
        if menu_result:
            intent_type, confidence, reason = menu_result
            logger.info(f"🍽️ Menu item: {intent_type.value} ({confidence:.2f}) - {reason}")
            return menu_result
        
        # No match found
        logger.info(f"❌ No rule match for: '{message[:50]}...'")
        return None
    
    def _pre_classify_guard(
        self,
        message: str,
        context: Dict
    ) -> Optional[Tuple[IntentType, float, str]]:
        """
        طبقة الحماية الأولى: كلمات مفتاحية عالية الأولوية
        """
        # Get high-priority rules (priority >= 9)
        high_priority_rules = [
            rule for rule in self.rules_loader.get_all_rules()
            if rule.priority >= 9
        ]
        
        for rule in high_priority_rules:
            # Check keywords first (faster than regex)
            keywords = rule.keywords.get('ar', []) + rule.keywords.get('en', [])
            
            for keyword in keywords:
                if keyword in message:
                    # Check exclusions
                    if rule.exclusions:
                        exclusions = rule.exclusions.get('ar', []) + rule.exclusions.get('en', [])
                        if any(excl in message for excl in exclusions):
                            continue
                    
                    # Check context requirements
                    if rule.context_required:
                        if not self._check_context_requirements(rule.context_required, context):
                            continue
                    
                    return (
                        rule.intent_type,
                        rule.confidence,
                        f"High-priority keyword: '{keyword}'"
                    )
        
        return None
    
    def _pattern_matching(
        self,
        message: str,
        context: Dict
    ) -> Optional[Tuple[IntentType, float, str]]:
        """
        مطابقة الأنماط (regex)
        """
        for rule in self.rules_loader.get_all_rules():
            if not rule.compiled_patterns:
                continue
            
            # Try all language patterns
            for lang, patterns in rule.compiled_patterns.items():
                for pattern in patterns:
                    if pattern.search(message):
                        # Check exclusions
                        if rule.exclusions:
                            exclusions = rule.exclusions.get('ar', []) + rule.exclusions.get('en', [])
                            if any(excl in message for excl in exclusions):
                                continue
                        
                        # Check context requirements
                        if rule.context_required:
                            if not self._check_context_requirements(rule.context_required, context):
                                continue
                        
                        # Calculate confidence based on context
                        confidence = self._calculate_confidence(rule, message, context)
                        
                        return (
                            rule.intent_type,
                            confidence,
                            f"Pattern match: {pattern.pattern[:50]}"
                        )
        
        return None
    
    def _special_patterns(
        self,
        message: str,
        context: Dict
    ) -> Optional[Tuple[IntentType, float, str]]:
        """
        أنماط خاصة (أرقام، رموز طلبات، إلخ)
        """
        # 1. Order number pattern
        order_pattern = self.rules_loader.special_patterns.get('order_number', {})
        if order_pattern:
            pattern = order_pattern.get('pattern', '')
            if pattern and re.search(pattern, message, re.IGNORECASE):
                return (IntentType.TRACK_ORDER, 0.95, "Order number detected")
        
        # 2. Number selection (1-20)
        if message.strip() in [str(i) for i in range(1, 21)]:
            if context.get('bot_presented_options') or context.get('last_bot_showed_options'):
                return (IntentType.SELECT_OPTION, 0.95, "Number selection")
        
        # 3. Arabic number selection
        if message.strip() in ['١', '٢', '٣', '٤', '٥', '٦', '٧', '٨', '٩', '١٠']:
            if context.get('bot_presented_options') or context.get('last_bot_showed_options'):
                return (IntentType.SELECT_OPTION, 0.95, "Arabic number selection")
        
        return None
    
    def _context_based_classification(
        self,
        message: str,
        context: Dict
    ) -> Optional[Tuple[IntentType, float, str]]:
        """
        تصنيف بناءً على السياق
        """
        # 1. Waiting for more items confirmation
        if context.get('waiting_for_more_items'):
            # Check for affirmative
            if message in ['نعم', 'yes', 'أيوه', 'أكيد', 'اه', 'آه', 'تمام']:
                return (IntentType.ORDER_FOOD, 0.90, "Add more items confirmation")
            # Check for negative
            elif message in ['لا', 'no', 'خلاص', 'كفاية']:
                return (IntentType.COMPLETE_ORDER, 0.90, "Complete order after 'add more' question")
        
        # 2. Bot asked a question
        if context.get('bot_asked_question'):
            # Simple yes/no
            if message in ['نعم', 'yes', 'أيوه', 'أكيد', 'اه']:
                return (IntentType.CONFIRM, 0.90, "Confirmation to bot question")
            elif message in ['لا', 'no', 'لأ']:
                return (IntentType.REJECT, 0.90, "Rejection to bot question")
        
        return None
    
    def _menu_item_detection(self, message: str) -> Optional[Tuple[IntentType, float, str]]:
        """
        كشف أصناف المنيو
        """
        if not self.menu_keywords:
            return None
        
        # Check if message contains menu item
        words = message.split()
        
        for word in words:
            if word in self.menu_keywords:
                return (IntentType.ORDER_FOOD, 0.85, f"Menu item detected: '{word}'")
        
        # Check full phrases
        for keyword in self.menu_keywords:
            if len(keyword) > 3 and keyword in message:
                return (IntentType.ORDER_FOOD, 0.85, f"Menu item detected: '{keyword}'")
        
        return None
    
    def _check_context_requirements(self, requirements: Dict, context: Dict) -> bool:
        """
        التحقق من متطلبات السياق
        """
        for key, expected_value in requirements.items():
            actual_value = context.get(key)
            
            if isinstance(expected_value, bool):
                if bool(actual_value) != expected_value:
                    return False
            elif actual_value != expected_value:
                return False
        
        return True
    
    def _calculate_confidence(self, rule: IntentRule, message: str, context: Dict) -> float:
        """
        حساب درجة الثقة
        """
        confidence = rule.confidence
        
        # Adjust based on context hints
        if rule.context_hints:
            for hint_key, hint_value in rule.context_hints.items():
                if context.get(hint_key) == hint_value:
                    confidence = min(confidence + 0.05, 1.0)
        
        # Reduce confidence for very long messages
        if len(message) > 100:
            confidence = max(confidence - 0.1, 0.5)
        
        return confidence
    
    def get_stats(self) -> Dict:
        """إحصائيات النظام"""
        return {
            'is_loaded': self._is_loaded,
            'menu_keywords_count': len(self.menu_keywords),
            **self.rules_loader.get_stats()
        }
    
    def reload_rules(self) -> bool:
        """إعادة تحميل القواعد"""
        logger.info("🔄 Reloading rules...")
        success = self.rules_loader.reload()
        if success:
            logger.info("✅ Rules reloaded successfully")
        return success


# Singleton instance
unified_classifier = UnifiedIntentClassifier()

