"""
Intent Rules Loader
محمل قواعد النوايا من ملفات YAML/JSON
"""

import yaml
import json
import re
from pathlib import Path
from typing import Dict, List, Optional, Tuple
from dataclasses import dataclass
from app.intents.intent_types import IntentType
from app.utils.logger import app_logger as logger


@dataclass
class IntentRule:
    """قاعدة نية واحدة"""
    name: str
    intent_type: IntentType
    priority: int
    confidence: float
    description: str
    patterns: Dict[str, List[str]]
    keywords: Dict[str, List[str]]
    exclusions: Optional[Dict[str, List[str]]] = None
    context_required: Optional[Dict] = None
    context_hints: Optional[Dict] = None
    examples: Optional[Dict[str, List[str]]] = None
    
    # Compiled patterns for faster matching
    compiled_patterns: Optional[Dict[str, List[re.Pattern]]] = None


class IntentRulesLoader:
    """
    محمل قواعد النوايا من الملفات
    
    يقرأ من:
    - data/intents_rules.yaml: القواعد والأنماط
    - data/intents_keywords.json: الكلمات المفتاحية الموسعة
    - data/intents_examples.json: أمثلة التدريب
    """
    
    def __init__(self, data_dir: str = "data"):
        self.data_dir = Path(data_dir)
        self.rules: List[IntentRule] = []
        self.keywords_db: Dict = {}
        self.examples_db: Dict = {}
        self.special_patterns: Dict = {}
        
        # Cache for faster lookups
        self._rules_by_intent: Dict[IntentType, IntentRule] = {}
        self._rules_by_priority: List[IntentRule] = []
    
    def load_all(self) -> bool:
        """
        تحميل جميع الملفات
        
        Returns:
            True إذا نجح التحميل، False إذا فشل
        """
        try:
            logger.info("📚 Loading intent rules from files...")
            
            # 1. Load rules
            rules_loaded = self._load_rules()
            if not rules_loaded:
                logger.error("❌ Failed to load intent rules")
                return False
            
            # 2. Load keywords database
            keywords_loaded = self._load_keywords()
            if not keywords_loaded:
                logger.warning("⚠️ Failed to load keywords database (optional)")
            
            # 3. Load examples
            examples_loaded = self._load_examples()
            if not examples_loaded:
                logger.warning("⚠️ Failed to load examples database (optional)")
            
            # 4. Build caches
            self._build_caches()
            
            logger.info(f"✅ Loaded {len(self.rules)} intent rules successfully")
            return True
            
        except Exception as e:
            logger.error(f"❌ Error loading intent rules: {str(e)}")
            return False
    
    def _load_rules(self) -> bool:
        """تحميل القواعد من YAML"""
        try:
            rules_file = self.data_dir / "intents_rules.yaml"
            
            if not rules_file.exists():
                logger.error(f"❌ Rules file not found: {rules_file}")
                return False
            
            with open(rules_file, 'r', encoding='utf-8') as f:
                data = yaml.safe_load(f)
            
            if not data:
                logger.error("❌ Empty rules file")
                return False
            
            # Extract special patterns
            self.special_patterns = data.get('special_patterns', {})
            
            # Parse rules
            rules_data = [item for item in data if isinstance(item, dict) and 'name' in item]
            
            for rule_data in rules_data:
                try:
                    rule = self._parse_rule(rule_data)
                    if rule:
                        self.rules.append(rule)
                except Exception as e:
                    logger.error(f"❌ Error parsing rule {rule_data.get('name')}: {str(e)}")
                    continue
            
            logger.info(f"✅ Loaded {len(self.rules)} rules from {rules_file}")
            return True
            
        except Exception as e:
            logger.error(f"❌ Error loading rules file: {str(e)}")
            return False
    
    def _parse_rule(self, rule_data: Dict) -> Optional[IntentRule]:
        """تحويل بيانات القاعدة إلى IntentRule"""
        try:
            name = rule_data['name']
            
            # Convert name to IntentType
            try:
                intent_type = IntentType[name]
            except KeyError:
                logger.warning(f"⚠️ Unknown intent type: {name}")
                return None
            
            # Compile patterns
            compiled_patterns = {}
            patterns = rule_data.get('patterns', {})
            
            for lang, pattern_list in patterns.items():
                compiled_patterns[lang] = []
                for pattern in pattern_list:
                    try:
                        compiled = re.compile(pattern, re.IGNORECASE)
                        compiled_patterns[lang].append(compiled)
                    except re.error as e:
                        logger.warning(f"⚠️ Invalid regex pattern '{pattern}': {str(e)}")
                        continue
            
            rule = IntentRule(
                name=name,
                intent_type=intent_type,
                priority=rule_data.get('priority', 5),
                confidence=rule_data.get('confidence', 0.9),
                description=rule_data.get('description', ''),
                patterns=patterns,
                keywords=rule_data.get('keywords', {}),
                exclusions=rule_data.get('exclusions'),
                context_required=rule_data.get('context_required'),
                context_hints=rule_data.get('context_hints'),
                examples=rule_data.get('examples'),
                compiled_patterns=compiled_patterns
            )
            
            return rule
            
        except Exception as e:
            logger.error(f"❌ Error parsing rule: {str(e)}")
            return None
    
    def _load_keywords(self) -> bool:
        """تحميل قاعدة الكلمات المفتاحية"""
        try:
            keywords_file = self.data_dir / "intents_keywords.json"
            
            if not keywords_file.exists():
                logger.warning(f"⚠️ Keywords file not found: {keywords_file}")
                return False
            
            with open(keywords_file, 'r', encoding='utf-8') as f:
                self.keywords_db = json.load(f)
            
            logger.info(f"✅ Loaded keywords database from {keywords_file}")
            return True
            
        except Exception as e:
            logger.error(f"❌ Error loading keywords file: {str(e)}")
            return False
    
    def _load_examples(self) -> bool:
        """تحميل أمثلة التدريب"""
        try:
            examples_file = self.data_dir / "intents_examples.json"
            
            if not examples_file.exists():
                logger.warning(f"⚠️ Examples file not found: {examples_file}")
                return False
            
            with open(examples_file, 'r', encoding='utf-8') as f:
                self.examples_db = json.load(f)
            
            logger.info(f"✅ Loaded examples database from {examples_file}")
            return True
            
        except Exception as e:
            logger.error(f"❌ Error loading examples file: {str(e)}")
            return False
    
    def _build_caches(self):
        """بناء الـ caches للبحث السريع"""
        # Build rules by intent
        self._rules_by_intent = {rule.intent_type: rule for rule in self.rules}
        
        # Build rules sorted by priority (highest first)
        self._rules_by_priority = sorted(self.rules, key=lambda r: r.priority, reverse=True)
        
        logger.info(f"✅ Built caches for {len(self.rules)} rules")
    
    def get_rule(self, intent_type: IntentType) -> Optional[IntentRule]:
        """الحصول على قاعدة نية معينة"""
        return self._rules_by_intent.get(intent_type)
    
    def get_all_rules(self) -> List[IntentRule]:
        """الحصول على جميع القواعد (مرتبة حسب الأولوية)"""
        return self._rules_by_priority
    
    def get_keywords_for_category(self, category: str, lang: str = 'ar') -> List[str]:
        """
        الحصول على كلمات مفتاحية لفئة معينة
        
        مثال:
        - category: "food_categories.appetizers"
        - category: "dialects.gulf.keywords.want"
        """
        try:
            parts = category.split('.')
            data = self.keywords_db
            
            for part in parts:
                data = data.get(part, {})
            
            if isinstance(data, dict):
                return data.get(lang, [])
            elif isinstance(data, list):
                return data
            else:
                return []
                
        except Exception as e:
            logger.error(f"❌ Error getting keywords for {category}: {str(e)}")
            return []
    
    def get_examples_for_intent(self, intent_name: str) -> List[Dict]:
        """الحصول على أمثلة لنية معينة"""
        try:
            intents = self.examples_db.get('intents', {})
            intent_data = intents.get(intent_name, {})
            return intent_data.get('examples', [])
        except Exception as e:
            logger.error(f"❌ Error getting examples for {intent_name}: {str(e)}")
            return []
    
    def reload(self) -> bool:
        """إعادة تحميل الملفات (للتحديث الحي)"""
        logger.info("🔄 Reloading intent rules...")
        self.rules.clear()
        self._rules_by_intent.clear()
        self._rules_by_priority.clear()
        return self.load_all()
    
    def get_stats(self) -> Dict:
        """إحصائيات النظام"""
        return {
            'total_rules': len(self.rules),
            'intents_covered': len(self._rules_by_intent),
            'has_keywords_db': bool(self.keywords_db),
            'has_examples_db': bool(self.examples_db),
            'total_keywords_categories': len(self.keywords_db.keys()) if self.keywords_db else 0,
            'total_examples': sum(
                len(intent_data.get('examples', []))
                for intent_data in self.examples_db.get('intents', {}).values()
            ) if self.examples_db else 0
        }


# Singleton instance
intent_rules_loader = IntentRulesLoader(data_dir="restaurant-chatbot-server/data")

