"""
Rule-Based Intent Classifier
تصنيف النية بالقواعد (بدون AI) - أسرع وأرخص وأدق
"""

from typing import Optional, Dict, List, Tuple
from app.intents.intent_types import IntentType
from app.intents.intent_guard import intent_guard
import re
import logging

logger = logging.getLogger(__name__)


class RuleBasedClassifier:
    """
    تصنيف النية بالقواعد البسيطة
    
    يغطي 90% من الحالات الشائعة:
    - طلب طعام
    - تأكيد/رفض
    - اختيار من قائمة
    - تحية/وداع
    - شكر
    - إلخ
    """
    
    def __init__(self):
        # تعريف الأنماط لكل نية
        self.patterns = self._build_patterns()
        
        # كلمات مفتاحية لأصناف المنيو (سيتم تحديثها من قاعدة البيانات)
        self.menu_keywords = set()
    
    def _build_patterns(self) -> Dict[IntentType, List[str]]:
        """بناء أنماط التصنيف"""
        return {
            # تأكيد
            IntentType.CONFIRM: [
                r'^نعم$', r'^yes$', r'^أيوه$', r'^أكيد$', r'^تمام$',
                r'^اه$', r'^آه$', r'^يب$', r'^yep$', r'^yeah$',
                r'^ok$', r'^okay$', r'^صح$', r'^ايوة$'
            ],
            
            # رفض
            IntentType.REJECT: [
                r'^لا$', r'^no$', r'^لأ$', r'^nope$',
                r'^مو$', r'^ما$', r'^not$', r'^لاء$'
            ],
            
            # إضافة المزيد (بعد سؤال "هل تريد إضافة المزيد؟")
            IntentType.ADD_MORE: [
                r'نعم.*إضافة', r'yes.*add', r'أريد.*المزيد',
                r'want.*more', r'بغيت.*زيادة', r'عايز.*كمان'
            ],
            
            # إكمال الطلب
            IntentType.COMPLETE_ORDER: [
                r'أكمل', r'complete', r'خلاص', r'كفاية', r'enough',
                r'إرسال.*موقع', r'send.*location', r'توصيل',
                r'delivery', r'كمل', r'finish', r'done'
            ],
            
            # تحية
            IntentType.GREETING: [
                r'^مرحبا', r'^hello', r'^hi$', r'^hey$', r'^السلام',
                r'^صباح', r'^مساء', r'^good morning', r'^good evening',
                r'^أهلا', r'^هلا', r'^يا هلا',
                r'كيف حالك', r'كيفك', r'شلونك', r'شخبارك', r'عامل ايه',
                r'how are you', r'how r u', r"what's up", r'whats up'
            ],
            
            # وداع
            IntentType.GOODBYE: [
                r'وداعا', r'bye', r'goodbye', r'مع السلامة',
                r'باي', r'تصبح على خير', r'good night'
            ],
            
            # شكر
            IntentType.THANKS: [
                r'شكرا', r'thanks', r'thank you', r'شكراً',
                r'مشكور', r'يعطيك العافية', r'الله يعافيك'
            ],
            
            # عرض المنيو
            IntentType.SHOW_MENU: [
                r'المنيو', r'القائمة', r'menu', r'عندكم.*ايش',
                r'what.*have', r'ايش.*عندكم', r'شو.*عندكم',
                r'عرض.*الطعام', r'show.*menu'
            ],
            
            # عرض السلة
            IntentType.SHOW_CART: [
                r'السلة', r'cart', r'طلبي', r'my order',
                r'ايش.*طلبت', r'what.*ordered', r'المجموع',
                r'total', r'كم.*السعر'
            ],

            # تعديل صنف في السلة 🆕
            IntentType.EDIT_ITEM: [
                r'خلي', r'خليها', r'خليه', r'خلي.*الصنف', r'خلي.*الطلب',
                r'غير', r'غيرها', r'غيره', r'عدل', r'عدلها', r'عدله',
                r'بدل', r'بدلها', r'بدله', r'اجعل', r'اجعلها', r'اجعله',
                r'change', r'modify', r'edit', r'make it', r'update', r'alter'
            ],

            # تتبع الطلب
            IntentType.TRACK_ORDER: [
                r'وين.*طلبي', r'where.*order', r'تتبع', r'track',
                r'متى.*يوصل', r'when.*arrive', r'كم.*باقي',
                r'how long', r'الطلب.*فين'
            ],
            
            # إلغاء الطلب
            IntentType.CANCEL_ORDER: [
                r'إلغاء', r'cancel', r'ألغي', r'مو.*عايز',
                r"don't want", r'بطل', r'stop'
            ],
            
            # التواصل مع خدمة العملاء 🆕
            IntentType.CONTACT_SUPPORT: [
                r'اكلم.*خدمة.*العملاء', r'اكلم.*حد.*خدمة', r'اتواصل.*الدعم',
                r'اكلم.*الادارة', r'اكلم.*مسؤول', r'رقم.*الدعم',
                r'كيف.*اكلم.*الدعم', r'اتواصل.*معكم', r'اريد.*رقمكم',
                r'اريد.*اكلم.*حد', r'محتاج.*اكلم.*حد', r'ابي.*اكلم.*حد',
                r'talk.*customer.*service', r'contact.*support', r'speak.*someone',
                r'customer.*care', r'call.*support', r'reach.*you'
            ],

            # أسئلة عامة (فروع، مواقع، أوقات عمل) 🆕
            IntentType.FAQ_INFO: [
                r'وين.*فروعكم', r'فين.*موقعكم', r'عندكم.*فرع',
                r'موقع.*المطعم', r'وينكم', r'اين.*فرعكم',
                r'ايش.*اوقات.*العمل', r'متى.*تفتحون', r'هل.*تفتحون.*اليوم',
                r'عندكم.*توصيل', r'توصلون.*لـ', r'كم.*رسوم.*التوصيل',
                r'where.*branch', r'location', r'branch.*near',
                r'opening.*hours', r'do.*you.*deliver', r'delivery.*fee'
            ],

            # مساعدة
            IntentType.HELP: [
                r'مساعدة', r'help', r'ساعدني',
                r'كيف.*اطلب', r'كيف.*استخدم', r'كيف.*اشتري',
                r'how.*order', r'how.*use', r'ازاي.*اطلب',
                r'شلون.*اطلب', r'وش.*اسوي'
            ],
        }
    
    async def update_menu_keywords(self, menu_items: List[Dict]):
        """تحديث كلمات المنيو من قاعدة البيانات"""
        self.menu_keywords.clear()
        
        for item in menu_items:
            # إضافة الاسم العربي
            name_ar = item.get('name_ar', '').lower()
            self.menu_keywords.add(name_ar)
            
            # إضافة الكلمات الفردية
            for word in name_ar.split():
                if len(word) > 2:  # تجاهل الكلمات القصيرة جداً
                    self.menu_keywords.add(word)
            
            # إضافة الاسم الإنجليزي
            name_en = item.get('name_en', '').lower()
            self.menu_keywords.add(name_en)
            
            for word in name_en.split():
                if len(word) > 2:
                    self.menu_keywords.add(word)
        
        logger.info(f"✅ Updated menu keywords: {len(self.menu_keywords)} keywords")
    
    def classify(
        self,
        message: str,
        context: Dict
    ) -> Optional[Tuple[IntentType, float]]:
        """
        تصنيف النية بالقواعد

        Args:
            message: رسالة المستخدم
            context: السياق الحالي

        Returns:
            (IntentType, confidence) أو None إذا لم يجد
        """
        message_lower = message.lower().strip()

        # 0. طبقة الحماية الأولى: فحص الكلمات المفتاحية ذات الأولوية العالية
        guard_result = intent_guard.pre_classify_guard(message)
        if guard_result:
            intent_type, reason = guard_result
            logger.info(f"🛡️ Intent guard pre-classified: {intent_type.value} | {reason}")
            return (intent_type, 0.95)  # ثقة عالية جداً

        # 1. فحص الأنماط المباشرة
        for intent, patterns in self.patterns.items():
            for pattern in patterns:
                if re.search(pattern, message_lower):
                    confidence = self._calculate_confidence(intent, message_lower, context)
                    logger.info(f"✅ Rule-based match: {intent.value} (confidence: {confidence:.2f})")
                    return (intent, confidence)
        
        # 2. فحص خاص: هل هو رقم؟ (اختيار من قائمة)
        if self._is_number_selection(message_lower, context):
            return (IntentType.SELECT_OPTION, 0.95)
        
        # 3. فحص خاص: هل هو رد على سؤال "هل تريد إضافة المزيد؟"
        if self._is_add_more_response(message_lower, context):
            if message_lower in ['نعم', 'yes', 'أيوه', 'أكيد', 'اه', 'آه']:
                return (IntentType.ORDER_FOOD, 0.90)  # يريد إضافة المزيد
            elif message_lower in ['لا', 'no', 'تمام', 'خلاص', 'كفاية']:
                return (IntentType.COMPLETE_ORDER, 0.90)  # يريد إكمال الطلب
        
        # 4. فحص خاص: هل يحتوي على اسم صنف من المنيو؟
        if self._contains_menu_item(message_lower):
            return (IntentType.ORDER_FOOD, 0.85)
        
        # 5. فحص خاص: هل هو رقم طلب؟
        if self._is_order_number(message_lower):
            return (IntentType.TRACK_ORDER, 0.90)
        
        # 6. لم يجد تطابق
        return None
    
    def _calculate_confidence(
        self,
        intent: IntentType,
        message: str,
        context: Dict
    ) -> float:
        """حساب درجة الثقة"""
        confidence = 0.9  # ثقة عالية للتطابق المباشر
        
        # زيادة الثقة إذا كان السياق يدعم النية
        if self._context_supports_intent(intent, context):
            confidence += 0.05
        
        # تقليل الثقة إذا كانت الرسالة طويلة جداً
        if len(message) > 100:
            confidence -= 0.1
        
        return min(confidence, 1.0)
    
    def _context_supports_intent(self, intent: IntentType, context: Dict) -> bool:
        """هل السياق يدعم هذه النية؟"""
        conversation_state = context.get('conversation_state', '')
        last_bot_question = context.get('last_bot_question', '')
        
        # أمثلة
        if intent == IntentType.COMPLETE_ORDER:
            # إذا كان هناك أصناف في السلة
            return len(context.get('items', [])) > 0
        
        if intent == IntentType.CONFIRM:
            # إذا كان البوت سأل سؤال نعم/لا
            return 'هل' in last_bot_question or 'do you' in last_bot_question.lower()
        
        return False
    
    def _is_number_selection(self, message: str, context: Dict) -> bool:
        """هل هو اختيار رقم من قائمة؟"""
        # فحص إذا كان رقم بسيط
        if not message.isdigit():
            return False
        
        number = int(message)
        
        # فحص إذا كان البوت عرض خيارات
        last_bot_showed_options = context.get('last_bot_showed_options', False)
        
        if last_bot_showed_options:
            # فحص إذا كان الرقم ضمن النطاق المعقول
            return 1 <= number <= 20
        
        return False
    
    def _is_add_more_response(self, message: str, context: Dict) -> bool:
        """هل هو رد على سؤال 'هل تريد إضافة المزيد؟'"""
        # 🆕 الطريقة الأولى: فحص الـ flag المباشر
        waiting_for_more_items = context.get('waiting_for_more_items', False)
        if waiting_for_more_items:
            logger.info(f"✅ Detected waiting_for_more_items flag in context")
            return True

        # الطريقة الثانية: فحص آخر رسالة من البوت (fallback)
        last_bot_message = context.get('last_bot_message', '')

        # فحص إذا كان البوت سأل عن إضافة المزيد
        add_more_triggers = [
            'هل تريد إضافة المزيد',
            'هل تريد اضافة المزيد',
            'would you like to add more',
            'أم إكمال الطلب',
            'أم إرسال موقعك'
        ]

        return any(trigger in last_bot_message for trigger in add_more_triggers)
    
    def _contains_menu_item(self, message: str) -> bool:
        """هل يحتوي على اسم صنف من المنيو؟"""
        if not self.menu_keywords:
            return False
        
        # فحص الكلمات
        words = message.split()
        
        for word in words:
            if word in self.menu_keywords:
                return True
        
        # فحص العبارات الكاملة
        for keyword in self.menu_keywords:
            if len(keyword) > 3 and keyword in message:
                return True
        
        return False
    
    def _is_order_number(self, message: str) -> bool:
        """هل هو رقم طلب؟"""
        # نمط رقم الطلب: ORD-XXXXXX
        pattern = r'ORD-\d{6}'
        return bool(re.search(pattern, message, re.IGNORECASE))
    
    def get_coverage_stats(self) -> Dict:
        """إحصائيات التغطية"""
        return {
            'total_intents': len(self.patterns),
            'total_patterns': sum(len(patterns) for patterns in self.patterns.values()),
            'menu_keywords': len(self.menu_keywords),
            'supported_intents': [intent.value for intent in self.patterns.keys()]
        }


# Singleton instance
rule_based_classifier = RuleBasedClassifier()

