"""
Intent Guard Layer - طبقة حماية لتصنيف النوايا
يمنع التصنيف الخاطئ للنوايا بناءً على كلمات مفتاحية محددة
"""

from typing import Optional, Tuple, Set
from app.intents.definitions import IntentType
from app.utils.logger import app_logger as logger
import re
import json
import os


class IntentGuard:
    """طبقة حماية ذكية لتصنيف النوايا"""
    
    # كلمات مفتاحية لكل نية (بالترتيب حسب الأولوية)
    INTENT_KEYWORDS = {
        # الأولوية 1: السؤال عن السعر (أعلى أولوية) - 200+ طريقة
        IntentType.ASK_PRICE: {
            'ar': [
                # الكلمات الأساسية
                'سعر', 'كم', 'ثمن', 'بكم', 'بكام', 'قيمة', 'تكلفة', 'كلفة',
                # مع الضمائر
                'سعره', 'سعرها', 'ثمنه', 'ثمنها', 'قيمته', 'قيمتها', 'كلفته', 'كلفتها',
                # مع "ال" التعريف
                'السعر', 'الثمن', 'القيمة', 'التكلفة',
                # الأفعال
                'يكلف', 'تكلف', 'يكلفني', 'تكلفني', 'يطلع', 'يصير', 'أدفع', 'بادفع',
                # الأفعال الاستفهامية
                'أعرف السعر', 'تقولي السعر', 'تعطيني السعر', 'تقدر تقولي',
                # اللهجات الخليجية
                'بجم', 'بچم', 'حق', 'حقه', 'حقها', 'شحالة', 'شحالته', 'قديش',
                # اللهجة المصرية
                'كام', 'بكام ده', 'كام ده',
                # اللهجة الشامية
                'قديش سعرو', 'شو سعرو', 'أديش', 'بدو', 'بده',
                # الصفات
                'غالي', 'غالية', 'رخيص', 'رخيصة', 'معقول', 'مناسب',
                # الصفات المقارنة
                'أرخص', 'أغلى', 'أوفر', 'فرق السعر',
                # الكلمات المساعدة
                'المبلغ', 'المجموع', 'الإجمالي', 'الحساب', 'الفاتورة', 'المطلوب',
                # الكميات
                'الواحد', 'الحبة', 'القطعة', 'الطبق', 'الصحن', 'الوجبة',
                # الأحجام
                'الصغير', 'الوسط', 'الكبير', 'العائلي'
            ],
            'en': [
                # Basic keywords
                'price', 'cost', 'how much', 'value', 'charge',
                # Adjectives
                'expensive', 'cheap', 'affordable', 'reasonable',
                # Totals
                'total', 'amount', 'bill', 'invoice',
                # Actions
                'pay', 'paying', 'payment',
                # Comparisons
                'cheaper', 'more expensive', 'difference',
                # Quantities
                'per piece', 'per item', 'each',
                # Sizes
                'small', 'medium', 'large', 'family size'
            ]
        },
        
        # الأولوية 2: السؤال عن المنيو
        IntentType.ASK_MENU: {
            'ar': ['منيو', 'المنيو', 'القائمة', 'قائمة', 'عندكم ايش', 'ايش عندكم', 'الأصناف', 'وريني'],
            'en': ['menu', 'what do you have', 'show me', 'list', 'items available']
        },
        
        # الأولوية 3: طلب طعام
        IntentType.ORDER_FOOD: {
            'ar': ['أريد', 'ابغى', 'ابي', 'طلب', 'اطلب', 'أوصي', 'اوصي', 'جهز', 'ضيف', 'زيد', 'حط'],
            'en': ['want', 'order', 'get me', 'i need', 'recommend', 'prepare']
        },
        
        # الأولوية 4: عرض السلة
        IntentType.VIEW_CART: {
            'ar': ['طلبي', 'السلة', 'المجموع', 'ايش طلبت', 'وش طلبت'],
            'en': ['my order', 'cart', 'basket', 'total', 'what did i order']
        },

        # الأولوية 5: حذف/مسح السلة
        IntentType.CLEAR_CART: {
            'ar': ['احذف السلة', 'امسح السلة', 'الغي السلة', 'حذف السلة', 'مسح السلة', 'إلغاء السلة', 'افرغ السلة', 'فرغ السلة'],
            'en': ['clear cart', 'delete cart', 'empty cart', 'remove all', 'clear basket', 'delete basket']
        }
    }
    
    # كلمات استثناء (إذا وُجدت مع كلمات أخرى، تلغي التصنيف)
    EXCLUSION_KEYWORDS = {
        IntentType.ASK_MENU: {
            # إذا وُجدت "سعر" مع "منيو"، لا تصنف كـ ask_menu
            'ar': ['سعر', 'كم', 'ثمن', 'بكم'],
            'en': ['price', 'cost', 'how much']
        }
    }
    
    def __init__(self):
        """تهيئة طبقة الحماية"""
        self._compile_patterns()
        self._load_menu_item_names()
    
    def _compile_patterns(self):
        """تجميع الأنماط للبحث السريع"""
        self.patterns = {}

        for intent_type, keywords in self.INTENT_KEYWORDS.items():
            self.patterns[intent_type] = {
                'ar': [re.compile(r'\b' + re.escape(kw) + r'\b', re.IGNORECASE) for kw in keywords['ar']],
                'en': [re.compile(r'\b' + re.escape(kw) + r'\b', re.IGNORECASE) for kw in keywords['en']]
            }

    def _load_menu_item_names(self):
        """تحميل أسماء الأصناف من الكاش"""
        self.menu_item_keywords: Set[str] = set()

        try:
            # Load from cache
            cache_path = os.path.join(
                os.path.dirname(os.path.dirname(os.path.dirname(__file__))),
                'data', 'cache', 'menu_cache.json'
            )

            if os.path.exists(cache_path):
                with open(cache_path, 'r', encoding='utf-8') as f:
                    menu_data = json.load(f)

                # Extract keywords from menu items
                for item in menu_data.get('items', []):
                    # Arabic name
                    name_ar = item.get('name_ar', '')
                    if name_ar:
                        # Add full name
                        self.menu_item_keywords.add(name_ar.lower())
                        # Add individual words (skip common words)
                        words = name_ar.split()
                        for word in words:
                            if len(word) > 2 and word not in ['مع', 'و', 'ب', 'من', 'في', 'على', 'رز', 'أبيض']:
                                self.menu_item_keywords.add(word.lower())

                    # English name
                    name_en = item.get('name_en', '')
                    if name_en:
                        # Add full name
                        self.menu_item_keywords.add(name_en.lower())
                        # Add individual words (skip common words)
                        words = name_en.lower().split()
                        for word in words:
                            if len(word) > 2 and word not in ['with', 'and', 'or', 'the', 'rice', 'white']:
                                self.menu_item_keywords.add(word)

                    # Tags
                    tags = item.get('tags', [])
                    for tag in tags:
                        if len(tag) > 2:
                            self.menu_item_keywords.add(tag.lower())

                logger.info(f"✅ Loaded {len(self.menu_item_keywords)} menu item keywords for intent guard")
            else:
                logger.warning(f"⚠️ Menu cache not found at {cache_path}")

        except Exception as e:
            logger.error(f"❌ Error loading menu item names: {e}")
            self.menu_item_keywords = set()

    def _contains_menu_item_name(self, message: str) -> bool:
        """
        التحقق من وجود اسم صنف في الرسالة

        Returns:
            True إذا كانت الرسالة تحتوي على اسم صنف
        """
        message_lower = message.lower().strip()

        # Check for exact matches or partial matches
        for keyword in self.menu_item_keywords:
            if keyword in message_lower:
                return True

        return False
    
    def pre_classify_guard(self, message: str) -> Optional[Tuple[IntentType, str]]:
        """
        طبقة حماية قبل التصنيف
        تتحقق من الكلمات المفتاحية وتصنف مباشرة إذا وُجدت
        
        Returns:
            Tuple[IntentType, reason] إذا تم التصنيف، None إذا لم يتم
        """
        
        message_lower = message.lower().strip()
        
        # التحقق من كل نية حسب الأولوية
        for intent_type in [IntentType.ASK_PRICE, IntentType.ASK_MENU, IntentType.ORDER_FOOD, IntentType.CLEAR_CART, IntentType.VIEW_CART]:
            if intent_type not in self.patterns:
                continue
            
            # البحث عن الكلمات المفتاحية
            matched_keywords = []
            
            for lang in ['ar', 'en']:
                for pattern in self.patterns[intent_type][lang]:
                    if pattern.search(message_lower):
                        matched_keywords.append(pattern.pattern.replace(r'\b', '').replace('\\', ''))
            
            if matched_keywords:
                # التحقق من كلمات الاستثناء
                if intent_type in self.EXCLUSION_KEYWORDS:
                    exclusion_found = False
                    
                    for lang in ['ar', 'en']:
                        for exclusion_kw in self.EXCLUSION_KEYWORDS[intent_type][lang]:
                            if exclusion_kw in message_lower:
                                exclusion_found = True
                                logger.warning(
                                    f"⚠️ Intent guard: Exclusion keyword '{exclusion_kw}' found, "
                                    f"skipping {intent_type.value}"
                                )
                                break
                        if exclusion_found:
                            break
                    
                    if exclusion_found:
                        continue
                
                # تم العثور على كلمات مفتاحية
                reason = f"Keywords detected: {', '.join(matched_keywords[:3])}"
                logger.info(
                    f"🛡️ Intent guard: Pre-classified as {intent_type.value} | {reason}"
                )
                return (intent_type, reason)
        
        return None
    
    def post_classify_guard(
        self,
        message: str,
        classified_intent: IntentType
    ) -> Tuple[IntentType, bool, Optional[str]]:
        """
        طبقة حماية بعد التصنيف
        تتحقق من صحة التصنيف وتصححه إذا لزم الأمر
        
        Returns:
            Tuple[IntentType, was_corrected, correction_reason]
        """
        
        message_lower = message.lower().strip()
        
        # قاعدة 1: إذا كانت الرسالة تحتوي على "سعر" ولكن صُنفت كـ ask_menu
        if classified_intent == IntentType.ASK_MENU:
            # استخدم regex للبحث عن كلمات كاملة فقط (لتجنب "عندكم" → "كم")
            # 200+ طريقة للسؤال عن السعر
            price_patterns_ar = [
                # الكلمات الأساسية
                r'\bسعر\b', r'\bكم\b', r'\bثمن\b', r'\bبكم\b', r'\bبكام\b',
                r'\bقيمة\b', r'\bتكلفة\b', r'\bكلفة\b',
                # مع الضمائر
                r'\bسعره\b', r'\bسعرها\b', r'\bثمنه\b', r'\bثمنها\b',
                r'\bقيمته\b', r'\bقيمتها\b', r'\bكلفته\b',
                # الأفعال
                r'\bيكلف\b', r'\bتكلف\b', r'\bيطلع\b', r'\bيصير\b', r'\bأدفع\b',
                # اللهجات
                r'\bبجم\b', r'\bبچم\b', r'\bحق\b', r'\bحقه\b', r'\bشحالة\b',
                r'\bقديش\b', r'\bكام\b', r'\bأديش\b', r'\bبدو\b',
                # الصفات
                r'\bغالي\b', r'\bرخيص\b', r'\bمعقول\b',
                # الكلمات المساعدة
                r'\bالمبلغ\b', r'\bالمجموع\b', r'\bالإجمالي\b', r'\bالحساب\b'
            ]
            price_patterns_en = [
                r'\bprice\b', r'\bcost\b', r'\bhow much\b', r'\bvalue\b',
                r'\bexpensive\b', r'\bcheap\b', r'\baffordable\b',
                r'\btotal\b', r'\bamount\b', r'\bpay\b', r'\bbill\b'
            ]

            for pattern in price_patterns_ar + price_patterns_en:
                if re.search(pattern, message_lower):
                    keyword = pattern.replace(r'\b', '')
                    logger.warning(
                        f"⚠️ Intent corrected: {classified_intent.value} → ask_price "
                        f"(keyword '{keyword}' detected)"
                    )
                    return (
                        IntentType.ASK_PRICE,
                        True,
                        f"Corrected from {classified_intent.value}: keyword '{keyword}' detected"
                    )
        
        # قاعدة 2: إذا كانت الرسالة تحتوي على "منيو" فقط (بدون "سعر")
        if classified_intent == IntentType.ASK_PRICE:
            menu_keywords_ar = ['منيو', 'المنيو', 'القائمة', 'قائمة']
            menu_keywords_en = ['menu', 'show menu']
            
            has_menu_keyword = any(kw in message_lower for kw in menu_keywords_ar + menu_keywords_en)
            has_price_keyword = any(
                kw in message_lower 
                for kw in ['سعر', 'كم', 'ثمن', 'بكم', 'price', 'cost', 'how much']
            )
            
            if has_menu_keyword and not has_price_keyword:
                logger.warning(
                    f"⚠️ Intent corrected: {classified_intent.value} → ask_menu "
                    f"(menu keyword detected without price keyword)"
                )
                return (
                    IntentType.ASK_MENU,
                    True,
                    f"Corrected from {classified_intent.value}: menu keyword without price"
                )
        
        # قاعدة 3: إذا كانت الرسالة قصيرة جدًا (<3 كلمات) وتحتوي على اسم صنف فقط
        words = message_lower.split()
        if len(words) <= 2 and classified_intent == IntentType.ASK_MENU:
            # قد تكون طلب طعام وليس طلب منيو
            logger.info(
                f"🛡️ Intent guard: Short message detected, keeping as {classified_intent.value}"
            )

        # قاعدة 4: إذا كانت الرسالة تحتوي على "ضيف/زيد/حط" + اسم صنف → ORDER_FOOD
        # حتى لو صُنفت خطأً كـ MODIFY_ORDER
        add_keywords = ['ضيف', 'زيد', 'حط', 'أضف', 'اضف', 'ضف', 'زود']
        has_add_keyword = any(kw in message_lower for kw in add_keywords)

        if classified_intent == IntentType.MODIFY_ORDER and has_add_keyword:
            logger.warning(
                f"⚠️ Intent corrected: {classified_intent.value} → order_food "
                f"(add keyword detected: user wants to add new item, not modify existing)"
            )
            return (
                IntentType.ORDER_FOOD,
                True,
                f"Corrected from {classified_intent.value}: add keyword detected"
            )

        # قاعدة 5: إذا كانت الرسالة تحتوي على اسم صنف، حولها إلى ORDER_FOOD
        # حتى لو كانت بصيغة سؤال (مثل: "عندكم سمك؟")
        # ⚠️ استثناء: إذا كانت الرسالة تحتوي على كلمات سعر، ابقها ASK_PRICE
        if classified_intent in [IntentType.ASK_MENU, IntentType.ASK_PRICE]:
            # التحقق من وجود كلمات سعر (يجب أن تتطابق مع القاعدة 1)
            price_keywords = [
                # الكلمات الأساسية
                'سعر', 'كم', 'ثمن', 'بكم', 'بكام', 'قيمة', 'تكلفة', 'كلفة',
                # مع الضمائر
                'سعره', 'سعرها', 'ثمنه', 'ثمنها', 'قيمته', 'قيمتها', 'كلفته',
                # الأفعال
                'يكلف', 'تكلف', 'يطلع', 'يصير', 'أدفع',
                # اللهجات (مهم جداً!)
                'بجم', 'بچم', 'حق', 'حقه', 'شحالة', 'قديش', 'كام', 'أديش', 'بدو',
                # الصفات
                'غالي', 'رخيص', 'معقول', 'أرخص', 'أغلى',
                # الكلمات المساعدة
                'المبلغ', 'المجموع', 'الإجمالي', 'الحساب', 'فرق السعر',
                # English
                'price', 'cost', 'how much', 'value', 'expensive', 'cheap', 'affordable',
                'total', 'amount', 'pay', 'bill'
            ]
            has_price_keyword = any(kw in message_lower for kw in price_keywords)

            # إذا كانت الرسالة تحتوي على كلمات سعر، ابقها ASK_PRICE
            if classified_intent == IntentType.ASK_PRICE and has_price_keyword:
                logger.info(
                    f"🛡️ Intent guard: Keeping as ask_price (price keywords detected)"
                )
                return (classified_intent, False, None)

            # إذا كانت الرسالة تحتوي على اسم صنف (بدون كلمات سعر)، حولها إلى ORDER_FOOD
            if self._contains_menu_item_name(message):
                logger.warning(
                    f"⚠️ Intent corrected: {classified_intent.value} → order_food "
                    f"(menu item name detected in message)"
                )
                return (
                    IntentType.ORDER_FOOD,
                    True,
                    f"Corrected from {classified_intent.value}: menu item name detected"
                )

        # لم يتم التصحيح
        return (classified_intent, False, None)
    
    def validate_intent_for_action(
        self,
        message: str,
        intent: IntentType,
        action: str
    ) -> bool:
        """
        التحقق من صحة النية قبل تنفيذ الإجراء

        Returns:
            True إذا كان الإجراء صحيح، False إذا يجب منعه
        """

        message_lower = message.lower().strip()

        # منع إرسال صور المنيو إذا كانت الرسالة تحتوي على "سعر"
        if intent == IntentType.ASK_MENU and action == "send_menu_images":
            price_keywords = ['سعر', 'كم', 'ثمن', 'بكم', 'price', 'cost', 'how much']

            for kw in price_keywords:
                if kw in message_lower:
                    logger.error(
                        f"❌ Action blocked: Cannot send menu images for message containing '{kw}'"
                    )
                    return False

        return True

    def detect_order_intent(self, message: str) -> Optional[Tuple[str, float]]:
        """
        كشف نية الطلب من الرسالة

        يتحقق من:
        1. وجود كلمات طلب صريحة (أريد، ابغى، طلب)
        2. وجود اسم صنف من المنيو
        3. وجود أرقام (قد تكون كمية)

        Args:
            message: رسالة المستخدم

        Returns:
            Tuple[intent_type, confidence] أو None
        """
        message_lower = message.lower().strip()

        # 1. التحقق من كلمات الطلب الصريحة
        order_keywords_ar = ['أريد', 'ابغى', 'ابي', 'بدي', 'عايز', 'طلب', 'اطلب', 'احجز', 'أوصي', 'اوصي', 'جهز', 'ضيف', 'زيد', 'حط']
        order_keywords_en = ['want', 'order', 'get me', 'i need', 'i would like', 'give me', 'recommend', 'prepare']

        has_order_keyword = False
        for kw in order_keywords_ar + order_keywords_en:
            if kw in message_lower:
                has_order_keyword = True
                break

        # 2. التحقق من وجود اسم صنف
        has_menu_item = self._contains_menu_item_name(message)

        # 3. التحقق من وجود أرقام (قد تكون كمية)
        has_number = bool(re.search(r'\d+|[٠-٩]+', message))

        # 4. حساب الثقة
        confidence = 0.0

        if has_order_keyword and has_menu_item:
            # كلمة طلب + اسم صنف = ثقة عالية جداً
            confidence = 0.95
            logger.info(f"🎯 Order intent detected: keyword + menu item")
        elif has_menu_item and has_number:
            # اسم صنف + رقم = ثقة عالية (مثل: "هريس 2")
            confidence = 0.90
            logger.info(f"🎯 Order intent detected: menu item + number")
        elif has_menu_item:
            # اسم صنف فقط = ثقة متوسطة (قد يكون سؤال)
            confidence = 0.75
            logger.info(f"🎯 Order intent detected: menu item only")
        elif has_order_keyword:
            # كلمة طلب فقط = ثقة منخفضة (يحتاج توضيح)
            confidence = 0.60
            logger.info(f"🎯 Order intent detected: order keyword only")

        if confidence > 0:
            return ('order_food', confidence)

        return None


# Singleton instance
intent_guard = IntentGuard()

