"""
Intent Definitions - تعريف جميع النوايا المدعومة
"""

from enum import Enum
from typing import List, Dict
from dataclasses import dataclass


class IntentType(str, Enum):
    """أنواع النوايا المدعومة"""
    
    # Order Management
    ORDER_FOOD = "order_food"              # طلب طعام
    VIEW_CART = "view_cart"                # عرض السلة
    CLEAR_CART = "clear_cart"              # حذف/مسح السلة 🆕
    MODIFY_ORDER = "modify_order"          # تعديل الطلب
    CANCEL_ORDER = "cancel_order"          # إلغاء الطلب
    CONFIRM_ORDER = "confirm_order"        # تأكيد الطلب
    
    # Information
    ASK_MENU = "ask_menu"                  # السؤال عن المنيو
    ASK_PRICE = "ask_price"                # السؤال عن السعر
    ASK_OFFERS = "ask_offers"              # السؤال عن العروض
    ASK_HOURS = "ask_hours"                # السؤال عن ساعات العمل
    ASK_LOCATION = "ask_location"          # السؤال عن الموقع
    
    # Order Tracking
    TRACK_ORDER = "track_order"            # تتبع الطلب
    REPEAT_ORDER = "repeat_order"          # إعادة طلب سابق
    REORDER = "reorder"                    # إعادة الطلب السابق
    
    # Checkout
    PROVIDE_ADDRESS = "provide_address"    # تقديم العنوان
    PROVIDE_PAYMENT = "provide_payment"    # تقديم طريقة الدفع
    
    # General
    SMALL_TALK = "small_talk"              # محادثة عامة
    ASK_CLARIFICATION = "ask_clarification" # طلب توضيح
    CONTACT_REQUEST = "contact_request"    # طلب التواصل


@dataclass
class IntentDefinition:
    """تعريف النية"""
    
    intent_type: IntentType
    name_ar: str
    name_en: str
    keywords_ar: List[str]
    keywords_en: List[str]
    examples_ar: List[str]
    examples_en: List[str]
    priority: int  # 1 = highest priority
    requires_context: bool = False


# تعريف جميع النوايا
INTENT_DEFINITIONS: Dict[IntentType, IntentDefinition] = {
    
    # Small Talk (أعلى أولوية)
    IntentType.SMALL_TALK: IntentDefinition(
        intent_type=IntentType.SMALL_TALK,
        name_ar="محادثة عامة",
        name_en="Small Talk",
        keywords_ar=["مرحبا", "السلام", "صباح", "مساء", "شكرا", "يعطيك", "مع السلامة", "باي", "كيف حالك"],
        keywords_en=["hello", "hi", "hey", "good morning", "good evening", "thanks", "thank you", "bye", "goodbye", "how are you"],
        examples_ar=["مرحبا", "السلام عليكم", "شكراً", "مع السلامة", "كيف حالك"],
        examples_en=["Hello", "Hi", "Thanks", "Bye", "How are you"],
        priority=1,
        requires_context=False
    ),
    
    # Order Food
    IntentType.ORDER_FOOD: IntentDefinition(
        intent_type=IntentType.ORDER_FOOD,
        name_ar="طلب طعام",
        name_en="Order Food",
        keywords_ar=["أريد", "عندكم", "طلب", "حمص", "سمك", "كبسة", "هريس", "شوربة", "سلطة"],
        keywords_en=["want", "order", "get", "hummus", "fish", "kabsa", "harees", "soup", "salad"],
        examples_ar=["أريد حمص", "عندكم سمك؟", "طلب كبسة", "2"],
        examples_en=["I want hummus", "Do you have fish?", "Order kabsa", "2"],
        priority=2,
        requires_context=True  # قد يكون رد على سؤال سابق
    ),
    
    # View Cart
    IntentType.VIEW_CART: IntentDefinition(
        intent_type=IntentType.VIEW_CART,
        name_ar="عرض السلة",
        name_en="View Cart",
        keywords_ar=["ما طلبي", "طلبي", "السلة", "المجموع", "كم", "ايش طلبت"],
        keywords_en=["my order", "cart", "basket", "total", "what did i order"],
        examples_ar=["ما طلبي", "ايش في السلة", "كم المجموع"],
        examples_en=["What's my order", "Show cart", "What's the total"],
        priority=2,
        requires_context=False
    ),

    # Clear Cart 🆕
    IntentType.CLEAR_CART: IntentDefinition(
        intent_type=IntentType.CLEAR_CART,
        name_ar="حذف/مسح السلة",
        name_en="Clear Cart",
        keywords_ar=["احذف السلة", "امسح السلة", "الغي السلة", "حذف السلة", "مسح السلة", "إلغاء السلة", "افرغ السلة"],
        keywords_en=["clear cart", "delete cart", "empty cart", "remove all", "clear basket"],
        examples_ar=["احذف السلة", "امسح السلة", "افرغ السلة"],
        examples_en=["Clear cart", "Delete cart", "Empty cart"],
        priority=2,
        requires_context=False
    ),
    
    # Ask Menu
    IntentType.ASK_MENU: IntentDefinition(
        intent_type=IntentType.ASK_MENU,
        name_ar="السؤال عن المنيو",
        name_en="Ask Menu",
        keywords_ar=["المنيو", "القائمة", "عندكم ايش", "ايش عندكم", "الأصناف"],
        keywords_en=["menu", "what do you have", "what's available", "items"],
        examples_ar=["عندكم ايش؟", "وريني المنيو", "ايش الأصناف"],
        examples_en=["What do you have?", "Show menu", "What items"],
        priority=2,
        requires_context=False
    ),

    # Ask Price
    IntentType.ASK_PRICE: IntentDefinition(
        intent_type=IntentType.ASK_PRICE,
        name_ar="السؤال عن السعر",
        name_en="Ask Price",
        keywords_ar=["سعر", "كم", "ثمن", "بكم", "قيمة", "تكلفة"],
        keywords_en=["price", "cost", "how much", "value"],
        examples_ar=["كم سعر الحمص؟", "ما سعر هريس اللحم؟", "بكم السمك؟"],
        examples_en=["How much is hummus?", "What's the price of fish?", "Cost of kabsa?"],
        priority=2,
        requires_context=False
    ),

    # Confirm Order
    IntentType.CONFIRM_ORDER: IntentDefinition(
        intent_type=IntentType.CONFIRM_ORDER,
        name_ar="تأكيد الطلب",
        name_en="Confirm Order",
        keywords_ar=["نعم", "أكمل", "تأكيد", "موافق", "اوكي", "اوك", "ok", "yes"],
        keywords_en=["yes", "confirm", "ok", "proceed", "continue", "yeah", "yep"],
        examples_ar=["نعم", "أكمل الطلب", "موافق", "yes"],
        examples_en=["Yes", "Confirm", "OK", "yes"],
        priority=3,
        requires_context=True  # يحتاج سياق
    ),
    
    # Provide Address
    IntentType.PROVIDE_ADDRESS: IntentDefinition(
        intent_type=IntentType.PROVIDE_ADDRESS,
        name_ar="تقديم العنوان",
        name_en="Provide Address",
        keywords_ar=["مسقط", "صلالة", "الخوير", "شارع", "بناية", "منزل"],
        keywords_en=["muscat", "salalah", "street", "building", "house", "address"],
        examples_ar=["مسقط، الخوير، شارع السلطان قابوس"],
        examples_en=["Muscat, Al Khuwair, Sultan Qaboos Street"],
        priority=3,
        requires_context=True
    ),
    
    # Track Order
    IntentType.TRACK_ORDER: IntentDefinition(
        intent_type=IntentType.TRACK_ORDER,
        name_ar="تتبع الطلب",
        name_en="Track Order",
        keywords_ar=["وين طلبي", "متى يصل", "كم باقي", "الطلب رقم"],
        keywords_en=["where is my order", "when will it arrive", "track order", "order number"],
        examples_ar=["وين طلبي؟", "متى يصل الطلب؟", "الطلب رقم 123"],
        examples_en=["Where is my order?", "When will it arrive?", "Order #123"],
        priority=2,
        requires_context=False
    ),
    
    # Modify Order
    IntentType.MODIFY_ORDER: IntentDefinition(
        intent_type=IntentType.MODIFY_ORDER,
        name_ar="تعديل الطلب",
        name_en="Modify Order",
        keywords_ar=["احذف", "غير", "عدل", "بدل", "ما أريد"],
        keywords_en=["remove", "delete", "change", "modify", "don't want"],
        examples_ar=["احذف الحمص", "غير الكمية", "ما أريد السمك"],
        examples_en=["Remove hummus", "Change quantity", "Don't want fish"],
        priority=2,
        requires_context=True
    ),
    
    # Cancel Order
    IntentType.CANCEL_ORDER: IntentDefinition(
        intent_type=IntentType.CANCEL_ORDER,
        name_ar="إلغاء الطلب",
        name_en="Cancel Order",
        keywords_ar=["إلغاء", "ألغي", "لا أريد", "cancel"],
        keywords_en=["cancel", "don't want", "abort"],
        examples_ar=["ألغي الطلب", "لا أريد شيء"],
        examples_en=["Cancel order", "Don't want anything"],
        priority=2,
        requires_context=False
    ),

    # Reorder
    IntentType.REORDER: IntentDefinition(
        intent_type=IntentType.REORDER,
        name_ar="إعادة الطلب",
        name_en="Reorder",
        keywords_ar=["أعد الطلب", "نفس الطلب", "كرر", "تكرار", "مثل السابق", "reorder"],
        keywords_en=["reorder", "repeat order", "same order", "same as before", "order again"],
        examples_ar=["أعد الطلب السابق", "نفس الطلب", "أريد تكرار الطلب"],
        examples_en=["Reorder", "Repeat my last order", "Same as before"],
        priority=2,
        requires_context=False
    ),

    # 🆕 Provide Payment
    IntentType.PROVIDE_PAYMENT: IntentDefinition(
        intent_type=IntentType.PROVIDE_PAYMENT,
        name_ar="تقديم طريقة الدفع",
        name_en="Provide Payment",
        keywords_ar=["نقد", "نقدي", "كاش", "cash", "بطاقة", "كريدت", "credit", "card", "الدفع عند الاستلام"],
        keywords_en=["cash", "credit", "card", "payment", "pay"],
        examples_ar=["نقدي", "كاش", "بطاقة", "الدفع عند الاستلام", "1", "2"],
        examples_en=["Cash", "Credit card", "1", "2"],
        priority=2,
        requires_context=True  # يتطلب أن تكون الجلسة في مرحلة الدفع
    ),

    # Ask Clarification
    IntentType.ASK_CLARIFICATION: IntentDefinition(
        intent_type=IntentType.ASK_CLARIFICATION,
        name_ar="طلب توضيح",
        name_en="Ask Clarification",
        keywords_ar=["ما فهمت", "وضح", "ايش يعني", "شرح"],
        keywords_en=["don't understand", "clarify", "what do you mean", "explain"],
        examples_ar=["ما فهمت", "وضح لي", "ايش يعني"],
        examples_en=["Don't understand", "Clarify", "What do you mean"],
        priority=1,
        requires_context=True
    ),
}


def get_intent_definition(intent_type: IntentType) -> IntentDefinition:
    """الحصول على تعريف النية"""
    return INTENT_DEFINITIONS.get(intent_type)


def get_all_intents() -> List[IntentDefinition]:
    """الحصول على جميع النوايا مرتبة حسب الأولوية"""
    return sorted(INTENT_DEFINITIONS.values(), key=lambda x: x.priority)


def get_intents_by_priority(priority: int) -> List[IntentDefinition]:
    """الحصول على النوايا حسب الأولوية"""
    return [intent for intent in INTENT_DEFINITIONS.values() if intent.priority == priority]

