"""
Intent Classifier - استخلاص النية من رسالة المستخدم
"""

from typing import Optional, Dict, List
from openai import AsyncOpenAI
import asyncio
from app.config import settings
from app.utils.logger import app_logger as logger
from app.intents.definitions import IntentType, get_all_intents, get_intent_definition
from app.intents.intent_guard import intent_guard


class IntentClassifier:
    """مصنف النوايا - يحدد نية المستخدم من رسالته"""
    
    def __init__(self):
        self.client = AsyncOpenAI(
            api_key=settings.OPENAI_API_KEY,
            timeout=15.0  # Timeout قصير للسرعة
        )
        self.model = "gpt-4o-mini"  # نموذج سريع
    
    async def classify_intent(
        self,
        user_message: str,
        context: Optional[Dict] = None,
        conversation_history: Optional[List[Dict]] = None
    ) -> IntentType:
        """
        تصنيف نية المستخدم
        
        Args:
            user_message: رسالة المستخدم
            context: السياق الحالي (السلة، المرحلة، إلخ)
            conversation_history: تاريخ المحادثة
            
        Returns:
            IntentType: نوع النية
        """
        try:
            logger.info(f"🎯 Classifying intent for: '{user_message[:50]}...'")

            # ✅ قاعدة مخصصة: إذا كان آخر سؤال من البوت "هل تريد إضافة المزيد..." والعميل أجاب بتأكيد قصير
            if self._is_affirmative_for_add_more(user_message, context, conversation_history):
                logger.info("✅ Detected 'add more' confirmation based on conversation context")
                return IntentType.ORDER_FOOD
            
            # بناء الـ prompt
            prompt = self._build_classification_prompt(
                user_message,
                context,
                conversation_history
            )
            
            # استدعاء OpenAI
            start_time = asyncio.get_event_loop().time()
            
            response = await asyncio.wait_for(
                self.client.chat.completions.create(
                    model=self.model,
                    messages=[
                        {"role": "system", "content": prompt},
                        {"role": "user", "content": user_message}
                    ],
                    temperature=0.1,  # دقة عالية
                    max_tokens=50  # رد قصير
                ),
                timeout=10.0
            )
            
            end_time = asyncio.get_event_loop().time()
            duration = end_time - start_time
            
            # استخراج النية
            intent_str = response.choices[0].message.content.strip().lower()
            intent_type = self._parse_intent(intent_str)

            logger.info(f"✅ Intent classified: {intent_type.value} ({duration:.2f}s)")

            # طبقة الحماية الثانية: التحقق بعد التصنيف
            corrected_intent, was_corrected, correction_reason = intent_guard.post_classify_guard(
                user_message,
                intent_type
            )

            if was_corrected:
                logger.warning(
                    f"⚠️ Intent corrected by guard: {intent_type.value} → {corrected_intent.value} | "
                    f"Reason: {correction_reason}"
                )
                return corrected_intent

            return intent_type
            
        except asyncio.TimeoutError:
            logger.error("❌ Intent classification timeout!")
            return self._fallback_intent_detection(user_message, context)
            
        except Exception as e:
            logger.error(f"❌ Error classifying intent: {str(e)}")
            return self._fallback_intent_detection(user_message, context)
    
    def _build_classification_prompt(
        self,
        user_message: str,
        context: Optional[Dict],
        conversation_history: Optional[List[Dict]]
    ) -> str:
        """بناء prompt لتصنيف النية"""
        
        # قائمة النوايا
        intents_list = []
        for intent_def in get_all_intents():
            intents_list.append(
                f"- {intent_def.intent_type.value}: {intent_def.name_ar} / {intent_def.name_en}"
            )
        
        prompt = f"""أنت مصنف نوايا ذكي. حدد نية المستخدم من رسالته.

## النوايا المتاحة:
{chr(10).join(intents_list)}

## السياق:
"""
        
        # إضافة السياق
        if context:
            stage = context.get('stage', 'items')
            cart_items = context.get('items', [])
            
            prompt += f"- المرحلة الحالية: {stage}\n"
            prompt += f"- عدد الأصناف في السلة: {len(cart_items)}\n"
        
        # إضافة آخر رسالة من البوت
        if conversation_history:
            last_bot_message = None
            for msg in reversed(conversation_history):
                if msg.get('role') == 'assistant':
                    last_bot_message = msg.get('content')
                    break
            
            if last_bot_message:
                prompt += f"- آخر رسالة من البوت: {last_bot_message[:100]}...\n"
        
        prompt += """
## التعليمات:
1. اقرأ رسالة المستخدم بعناية
2. افهم السياق (آخر رسالة من البوت)
3. حدد النية الأنسب
4. رد بكلمة واحدة فقط: اسم النية بالإنجليزي (مثل: order_food)

## أمثلة:
- "حمص" → order_food
- "ما طلبي" → view_cart
- "مرحبا" → small_talk
- "نعم" (بعد سؤال) → confirm_order
- "2" (بعد عرض خيارات) → order_food

**رد بكلمة واحدة فقط!**
"""
        
        return prompt
    
    def _parse_intent(self, intent_str: str) -> IntentType:
        """تحويل النص إلى IntentType"""
        
        # تنظيف النص
        intent_str = intent_str.strip().lower()
        
        # محاولة المطابقة المباشرة
        for intent_type in IntentType:
            if intent_type.value == intent_str:
                return intent_type
        
        # محاولة المطابقة الجزئية
        for intent_type in IntentType:
            if intent_type.value in intent_str or intent_str in intent_type.value:
                return intent_type
        
        # افتراضي
        logger.warning(f"⚠️ Unknown intent: {intent_str}, defaulting to ask_clarification")
        return IntentType.ASK_CLARIFICATION
    
    def _fallback_intent_detection(
        self,
        user_message: str,
        context: Optional[Dict]
    ) -> IntentType:
        """
        Fallback: تحديد النية بدون AI (keyword matching)
        """
        logger.info("🔄 Using fallback intent detection (keyword matching)")
        
        message_lower = user_message.lower()
        
        # Small Talk (أعلى أولوية)
        small_talk_keywords = ["مرحبا", "السلام", "صباح", "مساء", "شكرا", "باي", "hello", "hi", "thanks", "bye"]
        if any(kw in message_lower for kw in small_talk_keywords):
            return IntentType.SMALL_TALK
        
        # View Cart
        cart_keywords = ["ما طلبي", "طلبي", "السلة", "المجموع", "my order", "cart"]
        if any(kw in message_lower for kw in cart_keywords):
            return IntentType.VIEW_CART
        
        # Ask Menu
        menu_keywords = ["المنيو", "القائمة", "عندكم ايش", "menu", "what do you have"]
        if any(kw in message_lower for kw in menu_keywords):
            return IntentType.ASK_MENU
        
        # Track Order
        track_keywords = ["وين طلبي", "متى يصل", "track", "where is my order"]
        if any(kw in message_lower for kw in track_keywords):
            return IntentType.TRACK_ORDER
        
        # Cancel Order
        cancel_keywords = ["إلغاء", "ألغي", "cancel"]
        if any(kw in message_lower for kw in cancel_keywords):
            return IntentType.CANCEL_ORDER
        
        # Confirm Order (كلمات التأكيد في أي مرحلة)
        confirm_keywords = ["نعم", "أكمل", "موافق", "تمام", "ok", "yes", "confirm", "proceed", "اوكي", "اوك"]
        if any(kw in message_lower for kw in confirm_keywords):
            # إذا كان في مرحلة items وعنده أصناف في السلة → confirm
            if context and context.get('items') and len(context.get('items', [])) > 0:
                stage = context.get('stage', 'items')
                # في أي مرحلة (items, address, payment) → confirm
                if stage in ['items', 'address', 'payment']:
                    return IntentType.CONFIRM_ORDER
            # إذا كان في مرحلة address أو payment → confirm
            if context and context.get('stage') in ['address', 'payment']:
                return IntentType.CONFIRM_ORDER
        
        # Provide Address (في مرحلة العنوان)
        if context and context.get('stage') == 'address':
            address_keywords = ["مسقط", "صلالة", "شارع", "muscat", "street"]
            if any(kw in message_lower for kw in address_keywords):
                return IntentType.PROVIDE_ADDRESS
        
        # افتراضي: Order Food
        return IntentType.ORDER_FOOD

    def _is_affirmative_for_add_more(
        self,
        user_message: str,
        context: Optional[Dict],
        conversation_history: Optional[List[Dict]]
    ) -> bool:
        """
        Detect cases where the assistant just asked "هل تريد إضافة المزيد...?"
        and the customer replied with a short affirmative like "نعم".
        In this situation we should stay in the items stage instead of moving
        to confirmation.
        """
        if not conversation_history:
            return False

        # Only applies when we are still collecting items
        if not context or context.get('stage', 'items') != 'items':
            return False

        # Ensure there are already items in cart (otherwise no "add more" question)
        if not context.get('items'):
            return False

        normalized_reply = user_message.strip().lower().strip("!.؟?")
        if not normalized_reply:
            return False

        affirmative_replies = {
            "نعم",
            "ايه",
            "أيوه",
            "ايوه",
            "أكيد",
            "تمام",
            "طيب",
            "yes",
            "yeah",
            "yup",
            "yep",
            "sure",
            "ok",
            "okay",
            "okey",
            "تمامًا",
            "اوكي",
            "اوك",
        }

        if normalized_reply not in affirmative_replies:
            return False

        last_bot_message = None
        for msg in reversed(conversation_history):
            if msg.get('role') == 'assistant':
                last_bot_message = (msg.get('content') or "").lower()
                break

        if not last_bot_message:
            return False

        add_more_triggers = [
            "هل تريد إضافة المزيد",
            "هل تريد اضافة المزيد",
            "هل ترغب في إضافة المزيد",
            "هل ترغب في اضافة المزيد",
            "هل تريد إضافة صنف آخر",  # ✅ Single Question System
            "تبغى تضيف شي ثاني",  # ✅ Single Question System
            "would you like to add more",
            "do you want to add more",
            "would you like to add another item",  # ✅ Single Question System
            "anything else",  # ✅ Single Question System
        ]

        return any(trigger in last_bot_message for trigger in add_more_triggers)


# Singleton instance
intent_classifier = IntentClassifier()
