"""
Yes/No Response Handler - معالج ردود نعم/لا
يعالج ردود المستخدم بـ "نعم" أو "لا" بناءً على السياق
"""

from typing import Dict, Optional, List
from app.handlers.base_handler import BaseIntentHandler
from app.intents.intent_types import IntentType
from app.utils.question_context_manager import (
    question_context_manager,
    QuestionType
)
from app.utils.logger import app_logger as logger


class YesNoHandler(BaseIntentHandler):
    """معالج ردود نعم/لا"""
    
    # كلمات "نعم"
    YES_KEYWORDS_AR = ['نعم', 'ايوا', 'أيوه', 'اي', 'صح', 'تمام', 'موافق', 'اوكي']
    YES_KEYWORDS_EN = ['yes', 'yeah', 'yep', 'sure', 'ok', 'okay', 'fine']
    
    # كلمات "لا"
    NO_KEYWORDS_AR = ['لا', 'لأ', 'مو', 'ما', 'ماشي', 'مش', 'لاء']
    NO_KEYWORDS_EN = ['no', 'nope', 'nah', 'not']
    
    @property
    def intent_type(self) -> IntentType:
        return IntentType.CONFIRM  # أو يمكن إنشاء YES_NO intent جديد
    
    @property
    def prompt_template(self) -> str:
        return """أنت مساعد مطعم ذكي. مهمتك: معالجة ردود نعم/لا بناءً على السياق.

## القواعد:
1. تحقق من السؤال النشط في السياق
2. رد بناءً على نوع السؤال
3. امسح السياق بعد الرد
4. إذا لم يكن هناك سياق، اطلب التوضيح
"""
    
    def is_yes_response(self, message: str) -> bool:
        """
        التحقق من أن الرسالة "نعم"
        
        Args:
            message: رسالة المستخدم
            
        Returns:
            True إذا كانت "نعم"
        """
        message_lower = message.lower().strip()
        
        # تحقق من الكلمات العربية
        for keyword in self.YES_KEYWORDS_AR:
            if keyword in message_lower:
                return True
        
        # تحقق من الكلمات الإنجليزية
        for keyword in self.YES_KEYWORDS_EN:
            if keyword in message_lower:
                return True
        
        return False
    
    def is_no_response(self, message: str) -> bool:
        """
        التحقق من أن الرسالة "لا"
        
        Args:
            message: رسالة المستخدم
            
        Returns:
            True إذا كانت "لا"
        """
        message_lower = message.lower().strip()
        
        # تحقق من الكلمات العربية
        for keyword in self.NO_KEYWORDS_AR:
            if keyword in message_lower:
                return True
        
        # تحقق من الكلمات الإنجليزية
        for keyword in self.NO_KEYWORDS_EN:
            if keyword in message_lower:
                return True
        
        return False
    
    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """معالجة رد نعم/لا"""

        try:
            session = context.get('session', {})
            lang = self.get_language(context)

            # 1. التحقق من نوع الرد
            is_yes = self.is_yes_response(user_message)
            is_no = self.is_no_response(user_message)

            if not is_yes and not is_no:
                # ليس رد نعم/لا
                return None

            # 2. الحصول على السؤال النشط
            question_type = question_context_manager.get_question(session)

            if not question_type:
                # لا يوجد سؤال نشط - رد غامض
                logger.warning("⚠️ Yes/No response without active question")
                reply = question_context_manager.get_ambiguous_response(lang)

                return self.format_response(
                    intent="Ask Information",
                    reply=reply,
                    action=None,
                    data={'ambiguous_response': True}
                )

            # 3. معالجة الرد بناءً على نوع السؤال
            logger.info(
                f"{'✅' if is_yes else '❌'} Response to question: {question_type.value}"
            )

            if is_yes:
                reply = question_context_manager.get_yes_response(question_type, lang)
                action = self._get_yes_action(question_type)
                next_question = self._get_yes_next_question(question_type)
            else:
                reply = question_context_manager.get_no_response(question_type, lang)
                action = self._get_no_action(question_type)
                next_question = self._get_no_next_question(question_type)

            # 4. مسح السؤال من السياق
            question_context_manager.clear_question(session)

            # 5. تعيين السؤال التالي إذا كان موجوداً
            if next_question:
                question_context_manager.set_question(session, next_question)
                # إضافة رسالة السؤال للرد
                question_message = question_context_manager.get_question_message(next_question, lang)
                if question_message:
                    reply = f"{reply}\n\n{question_message}"

            # 6. إعداد البيانات الإضافية
            response_data = {
                'question_type': question_type.value,
                'response': 'yes' if is_yes else 'no',
                'next_question': next_question.value if next_question else None
            }

            # 🆕 إضافة last_question إلى response_data لحفظه في قاعدة البيانات
            if 'last_question' in session:
                response_data['last_question'] = session['last_question']

            # 7. إضافة session updates للحالات الخاصة
            session_updates = self._get_session_updates(question_type, is_yes)
            if session_updates:
                response_data.update(session_updates)
                # إذا كان هناك تحديثات للجلسة، نحتاج action
                if not action:
                    action = 'update_session'

            # 8. إرجاع الرد مع الإجراء
            return self.format_response(
                intent="Confirm" if is_yes else "Reject",
                reply=reply,
                action=action,
                data=response_data
            )

        except Exception as e:
            logger.error(f"❌ Error in yes/no handler: {str(e)}")
            return None
    
    def _get_yes_action(self, question_type: QuestionType) -> Optional[str]:
        """
        الحصول على الإجراء عند الرد بـ "نعم"
        
        Args:
            question_type: نوع السؤال
            
        Returns:
            اسم الإجراء أو None
        """
        action_map = {
            QuestionType.ADD_MORE_ITEMS: None,  # يبقى في نفس المرحلة
            QuestionType.CONFIRM_ORDER: 'confirm_order',
            QuestionType.CONFIRM_QUANTITY: None,
            QuestionType.ADD_EXTRAS: None,
            QuestionType.PROVIDE_ADDRESS: None,
            QuestionType.CONFIRM_ADDRESS: 'proceed_to_payment',
            QuestionType.CHOOSE_PAYMENT: None,
            QuestionType.CONFIRM_PAYMENT: 'create_order',
            QuestionType.CANCEL_ORDER: 'cancel_order',
            QuestionType.CLEAR_CART: 'clear_cart',
            QuestionType.REORDER: 'reorder_last_order'
        }
        
        return action_map.get(question_type)
    
    def _get_no_action(self, question_type: QuestionType) -> Optional[str]:
        """
        الحصول على الإجراء عند الرد بـ "لا"

        Args:
            question_type: نوع السؤال

        Returns:
            اسم الإجراء أو None
        """
        action_map = {
            QuestionType.ADD_MORE_ITEMS: None,  # سيتم تعيين سؤال جديد بدلاً من action
            QuestionType.CONFIRM_ORDER: None,
            QuestionType.CONFIRM_QUANTITY: None,
            QuestionType.ADD_EXTRAS: None,
            QuestionType.PROVIDE_ADDRESS: None,
            QuestionType.CONFIRM_ADDRESS: None,
            QuestionType.CHOOSE_PAYMENT: None,
            QuestionType.CONFIRM_PAYMENT: None,
            QuestionType.CANCEL_ORDER: None,
            QuestionType.CLEAR_CART: None,
            QuestionType.REORDER: None
        }

        return action_map.get(question_type)

    def _get_yes_next_question(self, question_type: QuestionType) -> Optional[QuestionType]:
        """
        الحصول على السؤال التالي عند الرد بـ "نعم"

        Args:
            question_type: نوع السؤال الحالي

        Returns:
            نوع السؤال التالي أو None
        """
        next_question_map = {
            QuestionType.ADD_MORE_ITEMS: None,  # يبقى في نفس المرحلة
            QuestionType.CONFIRM_ORDER: QuestionType.PROVIDE_ADDRESS,  # 🆕 الانتقال لطلب الموقع
            QuestionType.CONFIRM_QUANTITY: None,
            QuestionType.ADD_EXTRAS: None,
            QuestionType.PROVIDE_ADDRESS: QuestionType.CHOOSE_PAYMENT,  # 🆕 الانتقال لاختيار طريقة الدفع
            QuestionType.CONFIRM_ADDRESS: None,
            QuestionType.CHOOSE_PAYMENT: None,
            QuestionType.CONFIRM_PAYMENT: None,
            QuestionType.CANCEL_ORDER: None,
            QuestionType.CLEAR_CART: None,
            QuestionType.REORDER: None
        }

        return next_question_map.get(question_type)

    def _get_no_next_question(self, question_type: QuestionType) -> Optional[QuestionType]:
        """
        الحصول على السؤال التالي عند الرد بـ "لا"

        Args:
            question_type: نوع السؤال الحالي

        Returns:
            نوع السؤال التالي أو None
        """
        next_question_map = {
            QuestionType.ADD_MORE_ITEMS: QuestionType.CONFIRM_ORDER,  # الانتقال لتأكيد الطلب
            QuestionType.CONFIRM_ORDER: None,  # يبقى في نفس المرحلة
            QuestionType.CONFIRM_QUANTITY: None,  # سيطلب الكمية مرة أخرى
            QuestionType.ADD_EXTRAS: None,
            QuestionType.PROVIDE_ADDRESS: None,
            QuestionType.CONFIRM_ADDRESS: None,
            QuestionType.CHOOSE_PAYMENT: None,
            QuestionType.CONFIRM_PAYMENT: None,
            QuestionType.CANCEL_ORDER: None,
            QuestionType.CLEAR_CART: None,
            QuestionType.REORDER: None
        }

        return next_question_map.get(question_type)

    def _get_session_updates(self, question_type: QuestionType, is_yes: bool) -> Optional[Dict]:
        """
        الحصول على تحديثات الجلسة بناءً على نوع السؤال والرد

        Args:
            question_type: نوع السؤال
            is_yes: هل الرد "نعم"

        Returns:
            قاموس التحديثات أو None
        """
        # عند الرد بـ "نعم" على "هل تريد إضافة المزيد؟"
        # نحتاج مسح حالة الانتظار للسماح بإضافة أصناف جديدة
        if question_type == QuestionType.ADD_MORE_ITEMS and is_yes:
            logger.info("🔄 User wants to add more items, clearing waiting flags")
            return {
                'waiting_for_more_items': False,
                'waiting_for_item_choice': False,
                'waiting_for_quantity': False,
                'pending_item_choices': [],
                'selected_item_code': None,
                # 🆕 مسح السؤال من الجلسة لمنع طرح أسئلة إضافية
                'last_question': {
                    'type': QuestionType.NONE.value,
                    'asked_at': None
                }
            }

        # عند الرد بـ "لا" على "هل تريد إضافة المزيد؟"
        # الانتقال لتأكيد الطلب (يتم عبر next_question)
        if question_type == QuestionType.ADD_MORE_ITEMS and not is_yes:
            logger.info("✅ User finished adding items, ready to confirm order")
            return {
                'waiting_for_more_items': False
            }

        # 🔧 FIX: عند الرد بـ "نعم" على "هل تريد تأكيد الطلب؟"
        # نحتاج تحديث order_phase و stage للانتقال لمرحلة العنوان
        if question_type == QuestionType.CONFIRM_ORDER and is_yes:
            from app.state_machine.order_states import OrderPhase
            logger.info("✅ User confirmed order, moving to ADDRESS_INFO phase")
            return {
                'order_phase': OrderPhase.ADDRESS_INFO.value,
                'stage': 'address'
            }

        return None


# Singleton instance
yes_no_handler = YesNoHandler()

