"""
View Last Order Handler - معالج عرض آخر طلب
يعرض آخر طلب سابق للمستخدم
"""

from typing import Dict, Optional, List
from app.handlers.base_handler import BaseIntentHandler
from app.intents.definitions import IntentType
from app.utils.cart_formatter import cart_formatter
from app.utils.order_context_guard import order_context_guard
from app.utils.logger import app_logger as logger


class ViewLastOrderHandler(BaseIntentHandler):
    """معالج عرض آخر طلب"""
    
    @property
    def intent_type(self) -> IntentType:
        return IntentType.TRACK_ORDER  # نستخدم نفس النوع مؤقتاً
    
    @property
    def prompt_template(self) -> str:
        return """أنت مساعد مطعم ذكي. مهمتك: عرض آخر طلب سابق للعميل.

## القواعد:
1. اعرض تفاصيل آخر طلب
2. اعرض حالة الطلب
3. اسأل إذا كان يريد تكرار الطلب

## التنسيق:
- رد بالعربية إذا كان العميل يتحدث بالعربية
- رد بالإنجليزية إذا كان العميل يتحدث بالإنجليزية
- كن واضحاً ومنظماً
"""
    
    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """معالجة عرض آخر طلب"""
        
        try:
            logger.info(f"📜 Handling view last order")
            
            # 1. الحصول على البيانات
            session = context.get('session', {})
            lang = self.get_language(context)
            customer_id = context.get('customer_id')
            
            # 2. التحقق من الحارس
            guard_result = order_context_guard.can_view_last_order(session, 'last_order_intent')
            
            if not guard_result['allowed']:
                # ممنوع
                reason = guard_result['reason']
                message = guard_result.get(f'message_{lang}', guard_result['message_ar'])
                
                logger.warning(f"🚫 Blocked: {reason}")
                return self.format_response(
                    intent="Ask Information",
                    reply=message,
                    action=None,
                    data={'blocked': True, 'reason': reason}
                )
            
            # 3. جلب آخر طلب من قاعدة البيانات
            # (سيتم التنفيذ الفعلي في webhook.py)
            logger.info(f"📜 Fetching last order for customer: {customer_id}")
            
            # 4. إرسال action لجلب الطلب
            if lang == 'ar':
                reply = "📜 جاري البحث عن آخر طلب لك..."
            else:
                reply = "📜 Searching for your last order..."
            
            return self.format_response(
                intent="View Last Order",
                reply=reply,
                action="fetch_last_order",
                data={'customer_id': customer_id}
            )
            
        except Exception as e:
            logger.error(f"❌ Error in view last order handler: {str(e)}")
            return self._error_response(context)
    
    def format_last_order(
        self,
        order: Dict,
        lang: str = 'ar'
    ) -> str:
        """
        تنسيق عرض آخر طلب
        
        Args:
            order: بيانات الطلب
            lang: اللغة
            
        Returns:
            نص منسق
        """
        
        order_id = order.get('id', 'N/A')
        order_number = order.get('order_number', order_id)
        status = order.get('status', 'unknown')
        created_at = order.get('created_at', '')
        items = order.get('items', [])
        total = order.get('total', 0.0)
        
        if lang == 'ar':
            lines = [
                "📜 **آخر طلب سابق:**\n",
                f"🔢 رقم الطلب: #{order_number}",
                f"📅 التاريخ: {created_at}",
                f"📊 الحالة: {self._format_status_ar(status)}",
                ""
            ]
            
            # الأصناف
            if items:
                lines.append("📦 **الأصناف:**")
                for idx, item in enumerate(items, 1):
                    name = item.get('name_ar', item.get('name', 'صنف'))
                    quantity = item.get('quantity', 1)
                    price = item.get('price', 0.0)
                    line_total = quantity * price
                    lines.append(f"{idx}) {name} × {quantity} — {line_total:.2f} OMR")
                lines.append("")
            
            lines.append(f"💰 **الإجمالي: {total:.2f} OMR**")
            lines.append("")
            lines.append("💡 **هل تريد:**")
            lines.append("1️⃣ تكرار نفس الطلب")
            lines.append("2️⃣ العودة للسلة الحالية")
            
        else:
            lines = [
                "📜 **Last Previous Order:**\n",
                f"🔢 Order Number: #{order_number}",
                f"📅 Date: {created_at}",
                f"📊 Status: {self._format_status_en(status)}",
                ""
            ]
            
            # Items
            if items:
                lines.append("📦 **Items:**")
                for idx, item in enumerate(items, 1):
                    name = item.get('name_en', item.get('name', 'Item'))
                    quantity = item.get('quantity', 1)
                    price = item.get('price', 0.0)
                    line_total = quantity * price
                    lines.append(f"{idx}) {name} × {quantity} — {line_total:.2f} OMR")
                lines.append("")
            
            lines.append(f"💰 **Total: {total:.2f} OMR**")
            lines.append("")
            lines.append("💡 **Would you like to:**")
            lines.append("1️⃣ Repeat the same order")
            lines.append("2️⃣ Return to current cart")
        
        return "\n".join(lines)
    
    def _format_status_ar(self, status: str) -> str:
        """تنسيق الحالة بالعربي"""
        status_map = {
            'pending': '⏳ قيد الانتظار',
            'confirmed': '✅ مؤكد',
            'preparing': '👨‍🍳 قيد التحضير',
            'ready': '🎉 جاهز',
            'delivered': '✅ تم التوصيل',
            'cancelled': '❌ ملغي'
        }
        return status_map.get(status, status)
    
    def _format_status_en(self, status: str) -> str:
        """تنسيق الحالة بالإنجليزي"""
        status_map = {
            'pending': '⏳ Pending',
            'confirmed': '✅ Confirmed',
            'preparing': '👨‍🍳 Preparing',
            'ready': '🎉 Ready',
            'delivered': '✅ Delivered',
            'cancelled': '❌ Cancelled'
        }
        return status_map.get(status, status)
    
    def _error_response(self, context: Dict) -> Dict:
        """رد عند خطأ"""
        lang = self.get_language(context)
        
        if lang == 'ar':
            reply = "عذراً، حدث خطأ في عرض آخر طلب. من فضلك حاول مرة أخرى."
        else:
            reply = "Sorry, an error occurred while displaying the last order. Please try again."
        
        return self.format_response(
            intent="Ask Information",
            reply=reply,
            action=None,
            data=None
        )


# Singleton instance
view_last_order_handler = ViewLastOrderHandler()

