"""
View Cart Handler - معالج عرض السلة
يستخدم Cart Formatter الموحد + Order Intent Classifier
"""

from typing import Dict, Optional, List
from app.handlers.base_handler import BaseIntentHandler
from app.intents.definitions import IntentType
from app.utils.cart_formatter import cart_formatter
from app.utils.order_intent_classifier import order_intent_classifier
from app.utils.order_context_guard import order_context_guard
from app.utils.logger import app_logger as logger
from app.services.cart_service import cart_service


class ViewCartHandler(BaseIntentHandler):
    """معالج عرض السلة"""
    
    @property
    def intent_type(self) -> IntentType:
        return IntentType.VIEW_CART
    
    @property
    def prompt_template(self) -> str:
        return """أنت مساعد مطعم ذكي. مهمتك: عرض محتويات السلة للعميل.

## القواعد:
1. اعرض جميع الأصناف في السلة
2. اعرض المجموع الكلي
3. اسأل إذا كان يريد إضافة المزيد أو إكمال الطلب

## التنسيق:
- رد بالعربية إذا كان العميل يتحدث بالعربية
- رد بالإنجليزية إذا كان العميل يتحدث بالإنجليزية
- كن واضحاً ومنظماً
"""
    
    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """معالجة عرض السلة"""

        try:
            logger.info(f"🛍️ Handling view cart request")

            # 1. تصنيف النية - هل يقصد السلة الحالية أم الطلب السابق؟
            intent_result = order_intent_classifier.classify_order_intent(user_message)
            intent = intent_result['intent']
            confidence = intent_result['confidence']

            logger.info(f"🎯 Intent: {intent}, Confidence: {confidence}")

            # 2. الحصول على البيانات
            session = context.get('session', {})
            session_id = session.get('id')
            lang = self.get_language(context)

            # 🆕 Get cart items from cart_service (reads from draft_cart_items table)
            try:
                cart_summary = cart_service.get_cart_summary(session_id, lang)
                cart_items = cart_summary.get('items', [])
            except Exception as e:
                logger.error(f"❌ Error getting cart: {e}")
                cart_items = []

            # 3. التحقق من الحارس - هل يجب طلب توضيح؟
            should_clarify = order_context_guard.should_clarify(session, intent, confidence)

            if should_clarify:
                logger.warning(f"⚠️ Ambiguous intent - asking for clarification")
                reply = order_intent_classifier.get_clarification_message(lang)
                return self.format_response(
                    intent="Ask Information",
                    reply=reply,
                    action=None,
                    data={'awaiting_clarification': True}
                )

            # 4. إذا كان يقصد الطلب السابق - تحويل للـ handler المناسب
            if intent == 'last_order_intent':
                logger.info(f"📜 User wants last order, not current cart")

                # التحقق من الحارس
                guard_result = order_context_guard.can_view_last_order(session, intent)

                if not guard_result['allowed']:
                    # ممنوع - إرسال رسالة الحظر
                    reason = guard_result['reason']
                    message = guard_result.get(f'message_{lang}', guard_result['message_ar'])

                    logger.warning(f"🚫 Blocked: {reason}")
                    return self.format_response(
                        intent="Ask Information",
                        reply=message,
                        action=None,
                        data={'blocked': True, 'reason': reason}
                    )

                # مسموح - تحويل لـ view_last_order handler
                # (سيتم تنفيذه في المرحلة التالية)
                if lang == 'ar':
                    reply = "📜 سأعرض لك آخر طلب سابق..."
                else:
                    reply = "📜 I'll show you your last order..."

                return self.format_response(
                    intent="View Last Order",
                    reply=reply,
                    action="view_last_order",
                    data=None
                )

            # 5. عرض السلة الحالية
            logger.info(f"🛍️ Showing current cart")

            # التحقق إذا كانت السلة فارغة
            if not cart_items or len(cart_items) == 0:
                reply = cart_formatter.format_empty_cart_message(lang)
                return self.format_response(
                    intent="View Cart",
                    reply=reply,
                    action=None,
                    data=None
                )

            # 🆕 حساب الإجماليات من cart_service
            totals = cart_service.calculate_totals(session_id)

            # تنسيق السلة
            reply = cart_formatter.format_cart(cart_items, lang, totals)

            # 🆕 إضافة سؤال "هل تريد إضافة المزيد؟"
            from app.utils.question_context_manager import question_context_manager, QuestionType

            # إضافة السؤال للرد
            add_more_question = question_context_manager.get_question_message(
                QuestionType.ADD_MORE_ITEMS,
                lang
            )
            reply = f"{reply}\n\n{add_more_question}"

            return self.format_response(
                intent="View Cart",
                reply=reply,
                action="update_session",  # 🆕 نحتاج تحديث الجلسة
                data={
                    "totals": totals,
                    "item_count": len(cart_items),
                    "waiting_for_more_items": True,  # 🆕 تعيين flag
                    "last_question": {  # 🆕 حفظ السؤال
                        "type": QuestionType.ADD_MORE_ITEMS.value,
                        "asked_at": None  # سيتم تعيينه في webhook
                    }
                }
            )

        except Exception as e:
            logger.error(f"❌ Error in view cart handler: {str(e)}")
            return self._error_response(context)
    

    
    def _error_response(self, context: Dict) -> Dict:
        """رد عند خطأ"""
        lang = self.get_language(context)
        
        if lang == 'ar':
            reply = "عذراً، حدث خطأ في عرض السلة. من فضلك حاول مرة أخرى."
        else:
            reply = "Sorry, an error occurred while displaying the cart. Please try again."
        
        return self.format_response(
            intent="View Cart",
            reply=reply,
            action=None,
            data=None
        )


# Singleton instance
view_cart_handler = ViewCartHandler()

