"""
Track Order Handler - معالج تتبع الطلب
"""

from typing import Dict, Optional, List
from app.handlers.base_handler import BaseIntentHandler
from app.intents.definitions import IntentType
from app.utils.logger import app_logger as logger


class TrackOrderHandler(BaseIntentHandler):
    """معالج تتبع الطلب (وين طلبي، متى يصل)"""
    
    @property
    def intent_type(self) -> IntentType:
        return IntentType.TRACK_ORDER
    
    @property
    def prompt_template(self) -> str:
        return """أنت مساعد مطعم. المستخدم يريد تتبع طلبه."""
    
    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """معالجة تتبع الطلب"""
        
        try:
            logger.info(f"📦 Handling track order")
            
            lang = self.get_language(context)
            
            # التحقق: هل لديه طلب مفتوح في السلة؟
            cart_items = context.get('items', [])
            
            if cart_items:
                # لديه طلب مفتوح في السلة → عرض السلة
                logger.info("🛍️ User has open cart, showing cart")
                return self._show_open_cart(context, lang)
            else:
                # لا يوجد طلب مفتوح → عرض آخر طلب مكتمل
                logger.info("📋 No open cart, requesting last completed order")
                return self._request_last_order(lang)
            
        except Exception as e:
            logger.error(f"❌ Error in track order handler: {str(e)}")
            return self._error_response(context)
    
    def _show_open_cart(self, context: Dict, lang: str) -> Dict:
        """عرض الطلب المفتوح في السلة"""
        
        items = context.get('items', [])
        subtotal = context.get('subtotal', 0)
        delivery_fee = 0.500
        total = subtotal + delivery_fee
        
        # بناء قائمة الأصناف
        items_text = '\n'.join([
            f"• {item.get('name')} x{int(item.get('qty') or item.get('quantity', 1))}"
            for item in items
        ])
        
        if lang == 'ar':
            reply = f"""✅ **لديك طلب حالي قيد الإعداد:**

🛍️ **الأصناف:**
{items_text}

💰 **المجموع الفرعي:** {subtotal:.3f} ر.ع
🚚 **رسوم التوصيل:** {delivery_fee:.3f} ر.ع
💵 **الإجمالي:** {total:.3f} ر.ع

هل تريد إكمال هذا الطلب؟"""
        else:
            reply = f"""✅ **You have a current order in progress:**

🛍️ **Items:**
{items_text}

💰 **Subtotal:** {subtotal:.3f} OMR
🚚 **Delivery Fee:** {delivery_fee:.3f} OMR
💵 **Total:** {total:.3f} OMR

Would you like to complete this order?"""
        
        return self.format_response(
            intent="Track Order",
            reply=reply,
            action=None,
            data=None
        )
    
    def _request_last_order(self, lang: str) -> Dict:
        """طلب عرض آخر طلب مكتمل"""
        
        # هذا سيتم معالجته في webhook.py
        # نرسل action='track_order' بدون order_number
        # webhook.py سيبحث عن آخر طلب مكتمل
        
        if lang == 'ar':
            reply = "جاري البحث عن آخر طلب لك... ⏳"
        else:
            reply = "Searching for your last order... ⏳"
        
        return self.format_response(
            intent="Track Order",
            reply=reply,
            action="track_order",
            data={}  # بدون order_number → سيبحث عن آخر طلب
        )
    
    def _error_response(self, context: Dict) -> Dict:
        """رد عند خطأ"""
        lang = self.get_language(context)
        
        if lang == 'ar':
            reply = "عذراً، حدث خطأ. من فضلك حاول مرة أخرى."
        else:
            reply = "Sorry, an error occurred. Please try again."
        
        return self.format_response(
            intent="Track Order",
            reply=reply,
            action=None,
            data=None
        )


# Singleton instance
track_order_handler = TrackOrderHandler()

