"""
Small Talk Handler - معالج المحادثة العامة
"""

from typing import Dict, Optional, List
from app.handlers.base_handler import BaseIntentHandler
from app.intents.definitions import IntentType
from app.utils.logger import app_logger as logger
from app.responses.greeting_replies import get_smart_greeting_reply, detect_greeting_type


class SmallTalkHandler(BaseIntentHandler):
    """معالج المحادثة العامة (تحيات، شكر، وداع)"""
    
    @property
    def intent_type(self) -> IntentType:
        return IntentType.SMALL_TALK
    
    @property
    def prompt_template(self) -> str:
        return """أنت مساعد مطعم ودود. رد على المحادثة العامة بشكل مختصر ولطيف.

## أنواع المحادثة:
1. تحيات: "مرحبا"، "السلام عليكم"
2. شكر: "شكراً"، "يعطيك العافية"
3. وداع: "مع السلامة"، "باي"
4. سؤال عن الحال: "كيف حالك؟"

## القواعد:
- رد بنفس لغة المستخدم
- كن مختصراً (سطر أو سطرين)
- كن ودوداً ومحترفاً
"""
    
    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """معالجة المحادثة العامة"""

        try:
            logger.info(f"💬 Handling small talk: '{user_message}'")

            # تحديد نوع المحادثة
            message_lower = user_message.lower()
            lang = self.get_language(context)

            # تحيات - استخدام النظام الذكي الجديد
            # ملاحظة: التنظيف يحدث في webhook.py قبل الوصول هنا
            if any(word in message_lower for word in ["مرحبا", "السلام", "صباح", "مساء", "هلا", "أهلا", "حياك", "hello", "hi", "hey", "good morning", "good evening"]):
                # 🆕 استخدام النظام الذكي للردود
                greeting_type = detect_greeting_type(user_message)
                reply = get_smart_greeting_reply(user_message)
                logger.info(f"👋 Greeting detected: type={greeting_type}")

            # شكر
            elif any(word in message_lower for word in ["شكرا", "شكراً", "يعطيك", "جزاك", "thanks", "thank you"]):
                reply = self._thanks_response(lang)

            # وداع
            elif any(word in message_lower for word in ["مع السلامة", "باي", "وداعا", "bye", "goodbye"]):
                reply = self._goodbye_response(lang)

            # سؤال عن الحال
            elif any(word in message_lower for word in ["كيف حالك", "شخبارك", "how are you"]):
                reply = self._how_are_you_response(lang)

            # افتراضي
            else:
                reply = self._default_response(lang)

            return self.format_response(
                intent="Small Talk",
                reply=reply,
                action=None,
                data=None
            )

        except Exception as e:
            logger.error(f"❌ Error in small talk handler: {str(e)}")
            return self._error_response(context)
    
    def _greeting_response(self, lang: str, message: str = "") -> str:
        """
        رد على التحيات باستخدام النظام الذكي

        🆕 النظام الجديد:
        - يكتشف نوع التحية تلقائياً (سلام، مرحبا، هلا، صباح، مساء، إلخ)
        - يرد من نفس الفئة (context-aware)
        - أكثر من 100 رد متنوع
        """
        # 🆕 استخدام النظام الذكي للردود
        return get_smart_greeting_reply(message)
    
    def _thanks_response(self, lang: str) -> str:
        """رد على الشكر"""
        if lang == 'ar':
            return "العفو! 😊 سعداء بخدمتك"
        else:
            return "You're welcome! 😊 Happy to help"
    
    def _goodbye_response(self, lang: str) -> str:
        """رد على الوداع"""
        if lang == 'ar':
            return "مع السلامة! 🌟 نتطلع لخدمتك مرة أخرى"
        else:
            return "Goodbye! 🌟 Looking forward to serving you again"
    
    def _how_are_you_response(self, lang: str) -> str:
        """رد على السؤال عن الحال"""
        if lang == 'ar':
            return "بخير، شكراً! 😊 كيف يمكنني مساعدتك؟"
        else:
            return "I'm doing well, thanks! 😊 How can I help you?"
    
    def _default_response(self, lang: str) -> str:
        """رد افتراضي"""
        if lang == 'ar':
            return "أهلاً! كيف يمكنني مساعدتك؟"
        else:
            return "Hello! How can I help you?"
    
    def _error_response(self, context: Dict) -> Dict:
        """رد عند خطأ"""
        lang = self.get_language(context)
        
        if lang == 'ar':
            reply = "مرحباً! كيف يمكنني مساعدتك؟"
        else:
            reply = "Hello! How can I help you?"
        
        return self.format_response(
            intent="Small Talk",
            reply=reply,
            action=None,
            data=None
        )


# Singleton instance
small_talk_handler = SmallTalkHandler()

