"""
Reorder Handler - معالج إعادة الطلب
يسمح للمستخدم بتكرار آخر طلب سابق
"""

from typing import Dict, Optional, List
from app.handlers.base_handler import BaseIntentHandler
from app.intents.definitions import IntentType
from app.utils.cart_formatter import cart_formatter
from app.utils.order_context_guard import order_context_guard
from app.utils.logger import app_logger as logger


class ReorderHandler(BaseIntentHandler):
    """معالج إعادة الطلب"""
    
    @property
    def intent_type(self) -> IntentType:
        return IntentType.REORDER
    
    @property
    def prompt_template(self) -> str:
        return """أنت مساعد مطعم ذكي. مهمتك: إعادة آخر طلب سابق للعميل.

## القواعد:
1. جلب آخر طلب مؤكد
2. نسخ جميع الأصناف إلى سلة جديدة
3. عرض السلة الجديدة للعميل
4. السماح بالتعديل أو التأكيد

## التنسيق:
- رد بالعربية إذا كان العميل يتحدث بالعربية
- رد بالإنجليزية إذا كان العميل يتحدث بالإنجليزية
- كن واضحاً ومنظماً
"""
    
    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """معالجة إعادة الطلب"""
        
        try:
            logger.info(f"🔄 Handling reorder request")
            
            # 1. الحصول على البيانات
            session = context.get('session', {})
            lang = self.get_language(context)
            customer_id = context.get('customer_id')
            
            # 2. التحقق من الحارس - هل يمكن إعادة الطلب الآن؟
            guard_result = order_context_guard.can_view_last_order(session, 'reorder_intent')
            
            if not guard_result['allowed']:
                # ممنوع - لديه طلب حالي
                reason = guard_result['reason']
                message = guard_result.get(f'message_{lang}', guard_result['message_ar'])
                
                logger.warning(f"🚫 Reorder blocked: {reason}")
                return self.format_response(
                    intent="Ask Information",
                    reply=message,
                    action=None,
                    data={'blocked': True, 'reason': reason}
                )
            
            # 3. إرسال action لجلب آخر طلب وإعادته
            logger.info(f"🔄 Fetching last order for reorder: customer_id={customer_id}")
            
            if lang == 'ar':
                reply = "🔄 جاري إعادة طلبك السابق..."
            else:
                reply = "🔄 Reordering your last order..."
            
            return self.format_response(
                intent="Reorder",
                reply=reply,
                action="reorder_last_order",
                data={'customer_id': customer_id}
            )
            
        except Exception as e:
            logger.error(f"❌ Error in reorder handler: {str(e)}")
            return self._error_response(context)
    
    def format_reorder_success(
        self,
        items: List[Dict],
        totals: Dict,
        lang: str = 'ar'
    ) -> str:
        """
        تنسيق رسالة نجاح إعادة الطلب
        
        Args:
            items: الأصناف المعادة
            totals: الإجماليات
            lang: اللغة
            
        Returns:
            نص منسق
        """
        
        if lang == 'ar':
            header = """✅ **تمت إعادة طلبك السابق بنجاح!**

يمكنك الآن:
• تأكيد السلة والمتابعة للدفع
• تعديل الأصناف (إضافة أو حذف)

"""
        else:
            header = """✅ **Your last order has been reordered successfully!**

You can now:
• Confirm the cart and proceed to payment
• Modify items (add or remove)

"""
        
        # إضافة عرض السلة
        cart_display = cart_formatter.format_cart(items, lang, totals)
        
        return header + cart_display
    
    def format_no_previous_order(self, lang: str = 'ar') -> str:
        """
        رسالة عدم وجود طلب سابق
        
        Args:
            lang: اللغة
            
        Returns:
            رسالة منسقة
        """
        
        if lang == 'ar':
            return """❌ **لا يوجد طلب سابق لإعادة تنفيذه**

لم نجد أي طلبات سابقة مؤكدة لك.

💡 يمكنك:
• اختيار أصناف جديدة من المنيو
• اكتب "المنيو" لعرض القائمة"""
        else:
            return """❌ **No previous order to reorder**

We couldn't find any confirmed previous orders for you.

💡 You can:
• Choose new items from the menu
• Type "menu" to view the menu"""
    
    def format_reorder_with_unavailable_items(
        self,
        available_items: List[Dict],
        unavailable_items: List[Dict],
        totals: Dict,
        lang: str = 'ar'
    ) -> str:
        """
        تنسيق رسالة إعادة طلب مع أصناف غير متوفرة
        
        Args:
            available_items: الأصناف المتوفرة
            unavailable_items: الأصناف غير المتوفرة
            totals: الإجماليات
            lang: اللغة
            
        Returns:
            نص منسق
        """
        
        if lang == 'ar':
            header = f"""⚠️ **تمت إعادة طلبك السابق جزئياً**

⚠️ بعض الأصناف غير متوفرة حالياً ({len(unavailable_items)} صنف):
"""
            for item in unavailable_items:
                name = item.get('name_ar', item.get('name', 'صنف'))
                header += f"• {name}\n"
            
            header += "\n✅ تم إضافة الأصناف المتوفرة:\n\n"
        else:
            header = f"""⚠️ **Your last order has been partially reordered**

⚠️ Some items are currently unavailable ({len(unavailable_items)} items):
"""
            for item in unavailable_items:
                name = item.get('name_en', item.get('name', 'Item'))
                header += f"• {name}\n"
            
            header += "\n✅ Available items have been added:\n\n"
        
        # إضافة عرض السلة
        cart_display = cart_formatter.format_cart(available_items, lang, totals)
        
        return header + cart_display
    
    def _error_response(self, context: Dict) -> Dict:
        """رد عند خطأ"""
        lang = self.get_language(context)
        
        if lang == 'ar':
            reply = "عذراً، حدث خطأ في إعادة الطلب. من فضلك حاول مرة أخرى."
        else:
            reply = "Sorry, an error occurred while reordering. Please try again."
        
        return self.format_response(
            intent="Ask Information",
            reply=reply,
            action=None,
            data=None
        )


# Singleton instance
reorder_handler = ReorderHandler()

