"""
Handler Registry - تسجيل جميع معالجات النوايا
"""

from typing import Dict, Optional
from app.intents.definitions import IntentType
from app.handlers.base_handler import BaseIntentHandler
from app.handlers.order_food_handler import order_food_handler
from app.handlers.view_cart_handler import view_cart_handler
from app.handlers.view_last_order_handler import view_last_order_handler
from app.handlers.reorder_handler import reorder_handler
from app.handlers.yes_no_handler import yes_no_handler
from app.handlers.small_talk_handler import small_talk_handler
from app.handlers.ask_menu_handler import ask_menu_handler
from app.handlers.ask_price_handler import ask_price_handler
from app.handlers.confirm_order_handler import confirm_order_handler
from app.handlers.track_order_handler import track_order_handler
from app.handlers.provide_address_handler import provide_address_handler
from app.handlers.provide_payment_handler import provide_payment_handler
from app.handlers.edit_item_handler import EditItemHandler  # 🆕
from app.handlers.contact_support_handler import ContactSupportHandler  # 🆕
from app.handlers.faq_info_handler import FaqInfoHandler  # 🆕
from app.handlers.clear_cart_handler import clear_cart_handler  # 🆕
from app.utils.logger import app_logger as logger


class HandlerRegistry:
    """سجل معالجات النوايا"""
    
    def __init__(self):
        self._handlers: Dict[IntentType, BaseIntentHandler] = {}
        self._register_handlers()
    
    def _register_handlers(self):
        """تسجيل جميع المعالجات"""

        # General (أعلى أولوية)
        self.register(small_talk_handler)

        # Yes/No Responses (أولوية عالية جداً)
        self.register(yes_no_handler)

        # Order Management
        self.register(order_food_handler)
        self.register(view_cart_handler)
        self.register(view_last_order_handler)
        self.register(reorder_handler)
        self.register(confirm_order_handler)
        self.register(track_order_handler)

        # Cart Management 🆕
        edit_item_handler = EditItemHandler()
        self.register(edit_item_handler)
        self.register(clear_cart_handler)  # 🆕 حذف السلة

        # Checkout
        self.register(provide_address_handler)
        self.register(provide_payment_handler)

        # Information
        self.register(ask_menu_handler)
        self.register(ask_price_handler)

        # Customer Support & FAQ 🆕
        contact_support_handler = ContactSupportHandler()
        self.register(contact_support_handler)
        faq_info_handler = FaqInfoHandler()
        self.register(faq_info_handler)

        # TODO: إضافة باقي المعالجات
        # self.register(cancel_order_handler)
        # etc...

        logger.info(f"✅ Registered {len(self._handlers)} intent handlers")
    
    def register(self, handler: BaseIntentHandler):
        """تسجيل معالج"""
        self._handlers[handler.intent_type] = handler
        logger.debug(f"Registered handler for: {handler.intent_type.value}")
    
    def get_handler(self, intent_type: IntentType) -> Optional[BaseIntentHandler]:
        """الحصول على معالج حسب النية"""
        return self._handlers.get(intent_type)
    
    def has_handler(self, intent_type: IntentType) -> bool:
        """التحقق إذا كان هناك معالج لهذه النية"""
        return intent_type in self._handlers
    
    def get_all_handlers(self) -> Dict[IntentType, BaseIntentHandler]:
        """الحصول على جميع المعالجات"""
        return self._handlers.copy()


# Singleton instance
handler_registry = HandlerRegistry()

