"""
Provide Payment Handler - معالج تقديم طريقة الدفع
"""

from typing import Dict, Optional, List
from app.handlers.base_handler import BaseIntentHandler
from app.intents.definitions import IntentType
from app.utils.logger import app_logger as logger


class ProvidePaymentHandler(BaseIntentHandler):
    """معالج تقديم طريقة الدفع (نقداً، بطاقة ائتمان)"""
    
    @property
    def intent_type(self) -> IntentType:
        return IntentType.PROVIDE_PAYMENT
    
    @property
    def prompt_template(self) -> str:
        return """أنت مساعد مطعم. المستخدم يختار طريقة الدفع."""
    
    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """معالجة تقديم طريقة الدفع"""

        try:
            logger.info(f"💳 Handling provide payment")

            lang = self.get_language(context)
            stage = context.get('stage', 'items')
            session = context.get('session', {})
            last_question = session.get('last_question', '')

            # 🆕 حماية: إذا كانت الجلسة مكتملة بالفعل، لا تعالج الدفع
            session_status = session.get('status', 'active')
            if session_status == 'completed':
                logger.warning(f"⚠️ Session already completed, ignoring payment request")
                return None  # لا ترد بشيء

            # 🆕 التحقق: هل في مرحلة الدفع؟
            # نتحقق من stage أو last_question (للتوافق مع النظامين القديم والجديد)
            is_payment_stage = (stage == 'payment' or last_question == 'choose_payment')

            if not is_payment_stage:
                # ليس في مرحلة الدفع
                logger.warning(f"⚠️ User provided payment but not in payment stage (stage={stage}, last_question={last_question})")
                return self._not_expecting_payment(lang)

            logger.info(f"✅ Payment stage confirmed: stage={stage}, last_question={last_question}")

            # استخراج طريقة الدفع
            payment_method = self._extract_payment_method(user_message)

            if not payment_method:
                # لم نتمكن من تحديد طريقة الدفع
                logger.warning(f"⚠️ Could not extract payment method from: {user_message}")
                return self._ask_payment_again(lang)

            logger.info(f"💵 Payment method selected: {payment_method} by user message: {user_message}")

            # إنشاء الطلب
            return self._create_order(payment_method, lang)
            
        except Exception as e:
            logger.error(f"❌ Error in provide payment handler: {str(e)}")
            return self._error_response(context)
    
    def _extract_payment_method(self, message: str) -> Optional[str]:
        """استخراج طريقة الدفع من الرسالة"""

        message_lower = message.lower().strip()

        # Cash keywords (including single letter shortcuts)
        cash_keywords = ["نقد", "نقدي", "كاش", "cash", "1", "الدفع عند الاستلام", "ن"]
        if any(kw in message_lower for kw in cash_keywords):
            return "cash"

        # Credit card keywords (including single letter shortcuts)
        card_keywords = ["بطاقة", "بطاقه", "كريدت", "credit", "card", "2", "ب", "بي"]
        if any(kw in message_lower for kw in card_keywords):
            return "credit_card"

        return None
    
    def _create_order(self, payment_method: str, lang: str) -> Dict:
        """إنشاء الطلب"""
        
        if lang == 'ar':
            reply = "جاري إنشاء طلبك... ⏳"
        else:
            reply = "Creating your order... ⏳"
        
        return self.format_response(
            intent="Provide Payment",
            reply=reply,
            action="create_order",
            data={"payment_method": payment_method}
        )
    
    def _ask_payment_again(self, lang: str) -> Dict:
        """طلب طريقة الدفع مرة أخرى"""
        
        if lang == 'ar':
            reply = """💳 **اختر طريقة الدفع:**
1. الدفع عند الاستلام (نقداً)
2. بطاقة ائتمان

من فضلك اختر 1 أو 2."""
        else:
            reply = """💳 **Choose payment method:**
1. Cash on delivery
2. Credit card

Please choose 1 or 2."""
        
        return self.format_response(
            intent="Provide Payment",
            reply=reply,
            action=None,
            data=None
        )
    
    def _not_expecting_payment(self, lang: str) -> Dict:
        """رد عندما لا نتوقع طريقة دفع"""
        
        if lang == 'ar':
            reply = "شكراً! 💳\n\nلكن لست في مرحلة اختيار طريقة الدفع حالياً.\n\nماذا تريد أن تفعل؟"
        else:
            reply = "Thank you! 💳\n\nBut I'm not expecting a payment method right now.\n\nWhat would you like to do?"
        
        return self.format_response(
            intent="Provide Payment",
            reply=reply,
            action=None,
            data=None
        )
    
    def _error_response(self, context: Dict) -> Dict:
        """رد عند خطأ"""
        lang = self.get_language(context)
        
        if lang == 'ar':
            reply = "عذراً، حدث خطأ. من فضلك حاول مرة أخرى."
        else:
            reply = "Sorry, an error occurred. Please try again."
        
        return self.format_response(
            intent="Provide Payment",
            reply=reply,
            action=None,
            data=None
        )


# Singleton instance
provide_payment_handler = ProvidePaymentHandler()

