"""
Provide Address Handler - معالج تقديم العنوان والموقع
✅ يقبل فقط موقع GPS من واتساب
🚫 يرفض العنوان النصي أو الروابط بأسلوب لبق
📍 يدعم العربية والإنجليزية
"""

from typing import Dict, Optional, List
from app.handlers.base_handler import BaseIntentHandler
from app.intents.definitions import IntentType
from app.utils.location_extractor import location_extractor
from app.utils.logger import app_logger as logger
from app.state_machine.order_states import OrderPhase

RTL = "\u202B"


class ProvideAddressHandler(BaseIntentHandler):
    """معالج تقديم العنوان - يقبل الموقع فقط"""

    @property
    def intent_type(self) -> IntentType:
        return IntentType.PROVIDE_ADDRESS

    @property
    def prompt_template(self) -> str:
        return """المستخدم يرسل موقعه فقط (GPS). لا نقبل العنوان النصي."""

    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """معالجة رسالة الموقع"""
        try:
            logger.info("📍 Provide Address Handler: Processing location message")

            lang = self.get_language(context)
            session = context.get('session', {})
            session_id = session.get('id')

            current_phase = session.get('order_phase', OrderPhase.START.value)
            stage = session.get('stage', 'items')

            from app.utils.question_context_manager import question_context_manager, QuestionType
            last_question = question_context_manager.get_question(session)

            is_expecting_address = (
                current_phase == OrderPhase.ADDRESS_INFO.value or
                stage == 'address' or
                last_question == QuestionType.PROVIDE_ADDRESS
            )

            if not is_expecting_address:
                logger.warning(f"⚠️ Not expecting address (phase: {current_phase}, stage: {stage})")
                return self._not_expecting_address(lang)

            # ✅ قبول الموقع فقط من واتساب
            message_type = context.get('message_type', 'text')

            if message_type != 'location':
                # ❌ رفض أي نص أو رابط
                logger.warning("❌ Text or link received — expecting location only")
                reply = self._get_request_location_message(lang)
                return self.format_response(
                    intent="Location Not Found",
                    reply=reply,
                    action="request_location"
                )

            # ✅ معالجة موقع واتساب
            whatsapp_location = context.get('location_data', {})
            location_data = location_extractor.extract_from_whatsapp_location(whatsapp_location)

            if location_data and location_data.get('lat') and location_data.get('lng'):
                logger.info(f"✅ Location extracted: {location_data}")

                if session_id:
                    await self._save_location_to_session(session_id, location_data, context)

                reply = self._format_location_success_message(location_data, lang)

                return self.format_response(
                    intent="Location Received",
                    reply=reply,
                    action="update_session",
                    data={
                        "stage": "payment",
                        "order_phase": OrderPhase.PAYMENT.value,
                        "location": location_data,
                        "address": location_data.get('address') or f"Location: {location_data['lat']:.4f}, {location_data['lng']:.4f}"
                    }
                )

            else:
                logger.warning("❌ Invalid or empty location data")
                reply = self._get_request_location_message(lang)
                return self.format_response(
                    intent="Location Not Found",
                    reply=reply,
                    action="request_location"
                )

        except Exception as e:
            logger.error(f"❌ Error in provide address handler: {str(e)}")
            return self._error_response(context)

    async def _save_location_to_session(
        self,
        session_id: str,
        location_data: Dict,
        context: Dict
    ):
        """حفظ الموقع في الجلسة"""
        try:
            from app.services.session_manager import session_manager

            updates = {
                'location_lat': location_data.get('lat'),
                'location_lng': location_data.get('lng'),
                'location_link': location_data.get('link'),
                'address': location_data.get('address') or f"Location: {location_data['lat']:.4f}, {location_data['lng']:.4f}",
                'order_phase': OrderPhase.PAYMENT.value,
                'stage': 'payment'
            }

            await session_manager.update_session(session_id, updates)
            logger.info(f"✅ Location saved to session {session_id}")

        except Exception as e:
            logger.error(f"❌ Error saving location to session: {e}")

    def _format_location_success_message(self, location_data: Dict, lang: str) -> str:
        """تنسيق رسالة نجاح استلام الموقع"""
        if lang == "ar":
            reply = "✅ **تم استلام موقعك بنجاح!**\n\n"
            reply += f"📍 الإحداثيات: {location_data['lat']:.4f}, {location_data['lng']:.4f}\n\n"

            if location_data.get('link'):
                reply += f"🔗 [عرض على الخريطة]({location_data['link']})\n\n"

            reply += "💳 **اختر طريقة الدفع:**\n"
            reply += "1. الدفع عند الاستلام (نقداً)\n"
            reply += "2. بطاقة ائتمان"
        else:
            reply = "✅ **Location received successfully!**\n\n"
            reply += f"📍 Coordinates: {location_data['lat']:.4f}, {location_data['lng']:.4f}\n\n"

            if location_data.get('link'):
                reply += f"🔗 [View on map]({location_data['link']})\n\n"

            reply += "💳 **Choose payment method:**\n"
            reply += "1. Cash on delivery\n"
            reply += "2. Credit card"

        return reply

    def _get_request_location_message(self, lang: str) -> str:
        """رسالة تطلب من المستخدم إرسال الموقع فقط"""
        if lang == "ar":
            return (
                RTL + "📍 **من فضلك أرسل موقعك باستخدام خاصية المشاركة من واتساب.**\n\n"
                "**طريقة الإرسال:**\n"
                "1️⃣ اضغط على 📎 (مشبك الورق)\n"
                "2️⃣ اختر **Location (الموقع)**\n"
                "3️⃣ اضغط **Send your current location (إرسال موقعك الحالي)**"
            )
        else:
            return (
                "📍 **Please share your live location using WhatsApp's share feature.**\n\n"
                "**How to send:**\n"
                "1️⃣ Tap 📎 (attachment)\n"
                "2️⃣ Choose **Location**\n"
                "3️⃣ Tap **Send your current location**"
            )

    def _not_expecting_address(self, lang: str) -> Dict:
        """رد عندما لا نتوقع العنوان"""
        if lang == 'ar':
            reply = (
                RTL + "شكراً! 📍\n\n"
                "لكن لست في مرحلة إرسال الموقع حالياً.\n\n"
                "**لطلب توصيل:**\n"
                "1. اختر الأصناف التي تريدها\n"
                "2. قل \"اكمل الطلب\" أو \"نعم\"\n"
                "3. ثم أرسل موقعك\n\n"
                "هل تريد رؤية القائمة؟"
            )
        else:
            reply = (
                "Thank you! 📍\n\n"
                "But I'm not expecting a location right now.\n\n"
                "**To place an order:**\n"
                "1. Choose your items\n"
                "2. Say \"complete order\" or \"yes\"\n"
                "3. Then share your location\n\n"
                "Would you like to see the menu?"
            )

        return self.format_response(
            intent="Provide Address",
            reply=reply,
            action=None,
            data=None
        )

    def _error_response(self, context: Dict) -> Dict:
        """رد عند خطأ"""
        lang = self.get_language(context)
        if lang == 'ar':
            reply = RTL + "عذراً، حدث خطأ. من فضلك حاول مرة أخرى."
        else:
            reply = "Sorry, an error occurred. Please try again."

        return self.format_response(
            intent="Provide Address",
            reply=reply,
            action=None,
            data=None
        )


# Singleton instance
provide_address_handler = ProvideAddressHandler()
