"""
Order Food Handler - معالج طلب الطعام
"""

import asyncio
from typing import Dict, Optional, List
from app.handlers.base_handler import BaseIntentHandler
from app.intents.definitions import IntentType
from supabase import create_client
from app.config import settings
from app.utils.logger import app_logger as logger
from app.utils.message_formatter import (
    format_menu_items,
    format_single_item,
    format_no_results,
    format_item_added,
    format_quantity_request,
    format_invalid_selection,
    RTL
)

# Initialize Supabase client
supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)


class OrderFoodHandler(BaseIntentHandler):
    """معالج طلب الطعام"""
    
    @property
    def intent_type(self) -> IntentType:
        return IntentType.ORDER_FOOD
    
    @property
    def prompt_template(self) -> str:
        return """أنت مساعد مطعم ذكي. مهمتك: مساعدة العميل في طلب الطعام.

## القواعد:
1. ابحث عن الصنف في القائمة
2. إذا وجدت صنف واحد → اعرض معلوماته (الاسم، السعر، الوصف) واسأل عن الكمية
3. إذا وجدت عدة أصناف → اعرض الخيارات
4. إذا لم تجد → اعتذر واقترح بدائل

**مهم:** لا تضف الصنف تلقائياً! اعرض معلوماته أولاً واسأل عن الكمية

## التنسيق:
- رد بالعربية إذا كان العميل يتحدث بالعربية
- رد بالإنجليزية إذا كان العميل يتحدث بالإنجليزية
- كن مختصراً وواضحاً
- **ضع كل معلومة في سطر منفصل** (استخدم \n بين الأسطر)

## الرد:
رد بتنسيق JSON:
```json
{
  "action": "add_item" أو "show_options" أو "not_found",
  "item_code": "كود الصنف" (إذا وجدت صنف واحد),
  "options": [...] (إذا وجدت عدة أصناف),
  "reply": "الرد النصي للعميل (مع \n بين الأسطر)"
}
```

## مثال للرد عند عرض معلومات صنف واحد:
```
حمص 🍲

السعر: 1.200 ر.ع

كم الكمية المطلوبة؟
```

## مثال للرد عند إضافة صنف:
```
تم إضافة سمك مقلي مع رز قبولي إلى السلة ✅

السعر: 3.500 ر.ع

هل تريد إضافة صنف آخر؟
```

**ملاحظة مهمة:** اسأل سؤال واحد فقط في كل رسالة (Single Question System)
"""

    def _is_arabic_number(self, text: str) -> bool:
        """التحقق من أن النص يحتوي على أرقام عربية فقط"""
        arabic_numbers = '٠١٢٣٤٥٦٧٨٩'
        return all(c in arabic_numbers or c.isspace() for c in text.strip())

    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """معالجة طلب الطعام"""

        try:
            logger.info(f"🍽️ Handling order food: '{user_message}'")

            # 🆕 فحص طلب التكرار (Repeat Order Detection)
            repeat_order_response = await self._check_repeat_order(user_message, context)
            if repeat_order_response:
                logger.info("🔁 Repeat order detected")
                return repeat_order_response

            # ✅ التحقق من حالة الانتظار
            session = context.get('session', {})
            waiting_for_choice = session.get('waiting_for_item_choice', False)
            waiting_for_quantity = session.get('waiting_for_quantity', False)
            pending_items = session.get('pending_item_choices', [])
            selected_item_code = session.get('selected_item_code')

            # 🔍 سجل حالة الـ session للتشخيص
            logger.info(f"📊 Session state: waiting_for_choice={waiting_for_choice}, waiting_for_quantity={waiting_for_quantity}, pending_items={len(pending_items) if pending_items else 0}")

            # 🎯 تحديد نوع الإدخال: رقم أم نص
            is_number_input = user_message.strip().isdigit() or self._is_arabic_number(user_message.strip())

            # 🆕 تحسين: إذا كان الإدخال رقم بسيط (1-9) ولا توجد حالة انتظار، لا تعالجه كبحث
            is_simple_number = is_number_input and len(user_message.strip()) <= 2
            if is_simple_number and not waiting_for_choice and not waiting_for_quantity:
                logger.warning(f"⚠️ Simple number '{user_message}' without waiting state - treating as invalid")
                lang = self.get_language(context)
                if lang == 'ar':
                    reply = "من فضلك اختر صنف من القائمة أولاً، أو أخبرني ماذا تريد أن تطلب؟ 🍽️"
                else:
                    reply = "Please choose an item from the menu first, or tell me what you'd like to order? 🍽️"
                return self.format_response(
                    intent="Ask Information",
                    reply=reply,
                    action=None,
                    data=None
                )

            # ⚠️ الأولوية الأولى: إذا كان المستخدم في حالة انتظار الاختيار ودخل رقم
            # 🆕 تحسين: التأكد من عدم تفعيل waiting_for_quantity في نفس الوقت لتجنب التضارب
            if waiting_for_choice and not waiting_for_quantity and pending_items and is_number_input:
                logger.info(f"✅ Handling item selection from {len(pending_items)} pending items")
                return await self._handle_item_selection(user_message, context, pending_items)

            # الأولوية الثانية: إذا كان المستخدم في حالة انتظار الكمية ودخل رقم
            if waiting_for_quantity and selected_item_code and is_number_input:
                logger.info(f"✅ Handling quantity input for item: {selected_item_code}")
                return await self._handle_quantity_input(user_message, context, selected_item_code)

            # إذا كان المستخدم دخل نص (اسم صنف)، ابدأ بحث جديد وامسح الحالة القديمة
            if not is_number_input and (waiting_for_choice or waiting_for_quantity):
                logger.info(f"🔄 User entered text instead of number, starting new search and clearing old state")
                # امسح الحالة القديمة من الـ context
                waiting_for_choice = False
                waiting_for_quantity = False
                pending_items = []
                selected_item_code = None

                # 💾 Clear old state from session immediately
                await self._clear_selection_state(context)
                # استمر في البحث الجديد

            # 🆕 إذا كان المستخدم يطلب صنف جديد، امسح waiting_for_more_items flag
            # هذا يعني أنه اختار "نعم" وبدأ في طلب صنف جديد
            waiting_for_more_items = session.get('waiting_for_more_items', False)
            if waiting_for_more_items:
                logger.info(f"🔄 User is ordering new item, clearing waiting_for_more_items flag")

            # 🧠 استخراج الكمية واسم الصنف من النص
            from app.utils.text_helpers import extract_quantity_and_item
            quantity, item_name = extract_quantity_and_item(user_message)
            logger.info(f"📊 Extracted: quantity={quantity}, item='{item_name}'")

            # 🛡️ فلتر: منع البحث عن رسائل البوت نفسه
            bot_message_patterns = [
                "لدينا عدة خيارات",
                "أي منها تريد",
                "الأصناف المتاحة",
                "اختر رقم",
                "أرسل الرقم",
                "we have several",
                "which one would you like"
            ]

            if any(pattern in item_name.lower() for pattern in bot_message_patterns):
                logger.warning(f"⚠️ Detected bot message pattern in user input, treating as invalid")
                return self._handle_no_results(context, "")

            # حفظ الكمية في الـ context للاستخدام لاحقاً
            context['extracted_quantity'] = quantity

            # 1. البحث عن الأصناف المطابقة
            matching_items = await self._search_menu_items(item_name)

            # 2. معالجة النتائج حسب العدد
            if not matching_items:
                # لا توجد نتائج
                return self._handle_no_results(context)
            elif len(matching_items) == 1:
                # صنف واحد فقط
                extracted_qty = context.get('extracted_quantity', 1)

                # 🆕 التحقق إذا كان المستخدم ذكر رقم صريح (حتى لو كان 1)
                # نتحقق من وجود رقم في النص الأصلي
                import re
                has_explicit_number = bool(re.search(r'\d+|[٠-٩]+|واحد|وحدة|اثنين|اتنين|ثنتين|ثلاثة|تلاتة|أربعة|اربعة|خمسة|ستة|سبعة|ثمانية|تمانية|تسعة|عشرة', user_message, re.IGNORECASE))

                # إذا كانت الكمية محددة صراحة في النص (حتى لو كانت 1)، أضف الصنف مباشرة
                if has_explicit_number:
                    logger.info(f"✅ Quantity explicitly specified ({extracted_qty}), adding item directly")
                    return await self._add_item_to_cart(matching_items[0], extracted_qty, context)
                else:
                    # لم يذكر كمية - اسأل عن الكمية
                    logger.info(f"❓ No quantity specified, asking for quantity")
                    return self._handle_single_item(matching_items[0], context)
            else:
                # عدة أصناف - اعرض القائمة واطلب الاختيار
                return await self._handle_multiple_items(matching_items, context)
            
        except TimeoutError:
            logger.error("❌ Order food handler timeout")
            return self._timeout_response(context)
            
        except Exception as e:
            logger.error(f"❌ Error in order food handler: {str(e)}")
            return self._error_response(context)

    def _handle_no_results(self, context: Dict, query: str = "") -> Dict:
        """معالجة حالة عدم وجود نتائج"""
        lang = self.get_language(context)

        if lang == 'ar':
            reply = format_no_results(query) if query else RTL + "❌ عذراً، لم أجد أي صنف بهذا الاسم.\n\n" + RTL + "💡 أرسل \"المنيو\" لرؤية القائمة الكاملة. 📋"
        else:
            reply = "Sorry, I couldn't find any item with that name. 😔\n\nWould you like to see the full menu?"

        return self.format_response(
            intent="Ask Information",
            reply=reply,
            action=None,
            data=None
        )

    def _handle_single_item(self, item: Dict, context: Dict) -> Dict:
        """معالجة حالة وجود صنف واحد فقط"""
        lang = self.get_language(context)

        # تنسيق الرد باستخدام المنسق الجديد
        if lang == 'ar':
            reply = format_single_item(item)
        else:
            emoji = self._get_category_emoji(item.get('category', ''))
            name = item['name_en']
            price = float(item['price_omr'])
            reply = f"Found one item:\n\n{emoji} {name}\n\nPrice: {price:.3f} OMR\n\nHow many would you like?"

        # 💾 Store last item in context for price queries
        try:
            from app.services.context_manager import ContextManager
            chat_id = context.get('session', {}).get('chat_id')
            if chat_id:
                # We'll store it via the context manager in webhook.py
                # For now, just log it
                logger.info(f"💾 Single item found: {item.get('name_ar', item.get('name_en'))}")
        except Exception as e:
            logger.warning(f"⚠️ Could not store last item: {e}")

        # حفظ الصنف المختار في الـ session
        return self.format_response(
            intent="Ask Information",
            reply=reply,
            action="update_session",
            data={
                "selected_item_code": item['code'],
                "waiting_for_quantity": True,
                "waiting_for_more_items": False,  # 🆕 Clear flag
                "last_item_name": item.get('name_ar', item.get('name_en'))  # 💾 Store for context
            }
        )

    async def _handle_multiple_items(self, items: List[Dict], context: Dict) -> Dict:
        """معالجة حالة وجود عدة أصناف"""
        lang = self.get_language(context)

        # تحديد العدد الأقصى للعرض
        items_to_show = items[:5]  # أقصى 5 أصناف

        # بناء القائمة باستخدام المنسق الجديد
        if lang == 'ar':
            reply = format_menu_items(items_to_show)
        else:
            reply = "Found multiple items matching your request:\n\n"
            for idx, item in enumerate(items_to_show, 1):
                emoji = self._get_category_emoji(item.get('category', ''))
                name = item['name_en']
                price = float(item['price_omr'])
                reply += f"{idx}) {emoji} {name} — {price:.3f} OMR\n"
            reply += "\nPlease choose the item number."

        # 💾 Store first item name as context (user might ask about price later)
        first_item_name = items_to_show[0].get('name_ar', items_to_show[0].get('name_en'))
        logger.info(f"💾 Multiple items shown, first: {first_item_name}")

        # 🧠 حفظ الكمية المستخرجة للاستخدام بعد الاختيار
        extracted_qty = context.get('extracted_quantity', 1)

        # 🆕 حفظ العناصر الكاملة لتجنب الاستعلامات المكررة
        pending_items_full = {item['code']: item for item in items_to_show}

        # حفظ الأصناف في الـ session
        # Note: pending_quantity and pending_items_full will be stored in memory by webhook.py
        return self.format_response(
            intent="Ask Information",
            reply=reply,
            action="update_session",
            data={
                "waiting_for_item_choice": True,
                "pending_item_choices": [item['code'] for item in items_to_show],
                "pending_items_full": pending_items_full,  # 🆕 تخزين العناصر الكاملة
                "pending_quantity": extracted_qty,  # Will be stored in memory by webhook.py
                "waiting_for_more_items": False,  # 🆕 Clear flag
                "last_item_name": first_item_name  # 💾 Store for context
            }
        )

    async def _handle_item_selection(self, user_message: str, context: Dict, pending_items: List[str]) -> Dict:
        """معالجة اختيار المستخدم من القائمة"""
        lang = self.get_language(context)
        session = context.get('session', {})

        # محاولة استخراج الرقم من الرسالة
        import re
        numbers = re.findall(r'\d+', user_message)

        if not numbers:
            # لم يتم إدخال رقم
            if lang == 'ar':
                reply = format_invalid_selection()
            else:
                reply = "Please choose an item number from the list above. 🔢"

            return self.format_response(
                intent="Ask Information",
                reply=reply,
                action=None,
                data=None
            )

        # استخراج الرقم الأول
        choice_num = int(numbers[0])

        # التحقق من صحة الرقم
        if choice_num < 1 or choice_num > len(pending_items):
            if lang == 'ar':
                reply = format_invalid_selection()
            else:
                reply = f"Invalid number. Please choose a number from 1 to {len(pending_items)}."

            return self.format_response(
                intent="Ask Information",
                reply=reply,
                action=None,
                data=None
            )

        # الحصول على الصنف المختار
        selected_code = pending_items[choice_num - 1]

        # 🆕 محاولة الحصول على الصنف من الذاكرة المؤقتة أولاً
        pending_items_full = session.get('pending_items_full', {})
        selected_item = None

        if selected_code in pending_items_full:
            selected_item = pending_items_full[selected_code]
            logger.info(f"✅ Retrieved item from cache: {selected_code}")
        else:
            # Fallback: جلب من قاعدة البيانات
            logger.info(f"⚠️ Item not in cache, fetching from database: {selected_code}")
            try:
                result = supabase.table('menu_items').select('*').eq('code', selected_code).execute()

                if not result.data:
                    raise Exception(f"Item {selected_code} not found")

                selected_item = result.data[0]
            except Exception as e:
                logger.error(f"❌ Error fetching selected item: {e}")
                return self._error_response(context)

        # التحقق من وجود الصنف
        if not selected_item:
            logger.error(f"❌ Item {selected_code} not found")
            return self._error_response(context)

        try:

            # 🧠 التحقق من وجود كمية محفوظة مسبقاً
            pending_quantity = session.get('pending_quantity', 1)

            # إذا كانت الكمية محددة مسبقاً (> 1)، أضف الصنف مباشرة
            if pending_quantity > 1:
                logger.info(f"✅ Using saved quantity ({pending_quantity}), adding item directly")
                return await self._add_item_to_cart(selected_item, pending_quantity, context)

            # تنسيق رسالة التأكيد باستخدام المنسق الجديد
            name = selected_item['name_ar'] if lang == 'ar' else selected_item['name_en']

            if lang == 'ar':
                reply = format_quantity_request(name)
            else:
                emoji = self._get_category_emoji(selected_item.get('category', ''))
                price = float(selected_item['price_omr'])
                reply = f"Selected {emoji} {name}\n\nPrice: {price:.3f} OMR\n\nHow many would you like?"

            # 🆕 تخزين الصنف المختار في الذاكرة المؤقتة لاستخدامه في _handle_quantity_input
            selected_item_cache = {selected_code: selected_item}

            # تحديث الـ session
            return self.format_response(
                intent="Ask Information",
                reply=reply,
                action="update_session",
                data={
                    "selected_item_code": selected_code,
                    "selected_item_cache": selected_item_cache,  # 🆕 تخزين الصنف
                    "waiting_for_quantity": True,
                    "waiting_for_item_choice": False,
                    "pending_item_choices": [],
                    "pending_items_full": {},  # 🆕 مسح الذاكرة المؤقتة القديمة
                    "pending_quantity": 1  # Reset to 1, will be stored in memory by webhook.py
                }
            )

        except Exception as e:
            logger.error(f"❌ Error in item selection: {e}")
            return self._error_response(context)

    async def _add_item_to_cart(self, item: Dict, quantity: int, context: Dict) -> Dict:
        """
        إضافة صنف إلى السلة مباشرة

        Args:
            item: تفاصيل الصنف
            quantity: الكمية
            context: سياق المحادثة

        Returns:
            استجابة منسقة
        """
        lang = self.get_language(context)

        # تنسيق رسالة التأكيد
        name = item['name_ar'] if lang == 'ar' else item['name_en']
        price = float(item['price_omr'])

        if lang == 'ar':
            reply = format_item_added(name, quantity)
        else:
            emoji = self._get_category_emoji(item.get('category', ''))
            total = price * quantity
            reply = f"Added {emoji} {name} to cart ✅\n\n"
            reply += f"Quantity: {quantity}\n"
            reply += f"Price: {price:.3f} OMR\n"
            reply += f"Total: {total:.3f} OMR\n\n"
            reply += "Would you like to add another item?"

        # إضافة الصنف إلى السلة
        return self.format_response(
            intent="New Order - Items",
            reply=reply,
            action="add_item",
            data={
                "code": item['code'],
                "name": name,
                "name_ar": item.get('name_ar', name),
                "name_en": item.get('name_en', name),
                "quantity": quantity,
                "price": price,
                # 🆕 مسح جميع حالات الانتظار (بما في ذلك waiting_for_more_items)
                "selected_item_code": None,
                "waiting_for_quantity": False,
                "waiting_for_item_choice": False,
                "waiting_for_more_items": False,  # 🆕 إضافة
                "pending_item_choices": [],
                "pending_quantity": 1  # Reset, will be stored in memory by webhook.py
            }
        )

    async def _handle_quantity_input(self, user_message: str, context: Dict, item_code: str) -> Dict:
        """معالجة إدخال الكمية"""
        lang = self.get_language(context)
        session = context.get('session', {})

        # محاولة استخراج الرقم من الرسالة
        import re
        numbers = re.findall(r'\d+', user_message)

        if not numbers:
            # لم يتم إدخال رقم
            if lang == 'ar':
                reply = "من فضلك أدخل الكمية المطلوبة (رقم). 🔢"
            else:
                reply = "Please enter the quantity (number). 🔢"

            return self.format_response(
                intent="Ask Information",
                reply=reply,
                action=None,
                data=None
            )

        # استخراج الرقم الأول
        quantity = int(numbers[0])

        # التحقق من صحة الكمية
        if quantity < 1 or quantity > 99:
            if lang == 'ar':
                reply = "الكمية غير صحيحة. من فضلك أدخل رقمًا من 1 إلى 99."
            else:
                reply = "Invalid quantity. Please enter a number from 1 to 99."

            return self.format_response(
                intent="Ask Information",
                reply=reply,
                action=None,
                data=None
            )

        # 🆕 محاولة الحصول على الصنف من الذاكرة المؤقتة أولاً
        selected_item_cache = session.get('selected_item_cache', {})
        item = None

        if item_code in selected_item_cache:
            item = selected_item_cache[item_code]
            logger.info(f"✅ Retrieved item from cache for quantity input: {item_code}")
        else:
            # Fallback: جلب من قاعدة البيانات
            logger.info(f"⚠️ Item not in cache, fetching from database: {item_code}")
            try:
                result = supabase.table('menu_items').select('*').eq('code', item_code).execute()

                if not result.data:
                    raise Exception(f"Item {item_code} not found")

                item = result.data[0]
            except Exception as e:
                logger.error(f"❌ Error fetching item for quantity: {e}")
                return self._error_response(context)

        # التحقق من وجود الصنف
        if not item:
            logger.error(f"❌ Item {item_code} not found")
            return self._error_response(context)

        try:

            # تنسيق رسالة التأكيد باستخدام المنسق الجديد
            name = item['name_ar'] if lang == 'ar' else item['name_en']
            price = float(item['price_omr'])  # ✅ تعريف price هنا لاستخدامه في كلا الحالتين

            if lang == 'ar':
                reply = format_item_added(name, quantity)
            else:
                emoji = self._get_category_emoji(item.get('category', ''))
                total = price * quantity
                reply = f"Added {emoji} {name} to cart ✅\n\n"
                reply += f"Quantity: {quantity}\n"
                reply += f"Price: {price:.3f} OMR\n"
                reply += f"Total: {total:.3f} OMR\n\n"
                reply += "Would you like to add another item?"

            # إضافة الصنف إلى السلة
            return self.format_response(
                intent="New Order - Items",
                reply=reply,
                action="add_item",
                data={
                    "code": item['code'],
                    "name": name,
                    "name_ar": item.get('name_ar', name),
                    "name_en": item.get('name_en', name),
                    "quantity": quantity,
                    "price": price,
                    # 🆕 مسح جميع حالات الانتظار والذاكرة المؤقتة
                    "selected_item_code": None,
                    "selected_item_cache": {},  # 🆕 مسح الذاكرة المؤقتة
                    "waiting_for_quantity": False,
                    "waiting_for_item_choice": False,
                    "waiting_for_more_items": False,
                    "pending_item_choices": [],
                    "pending_items_full": {}  # 🆕 مسح الذاكرة المؤقتة
                }
            )

        except Exception as e:
            logger.error(f"❌ Error in quantity input: {e}")
            return self._error_response(context)

    def _get_category_emoji(self, category: str) -> str:
        """الحصول على الإيموجي المناسب للفئة"""
        emoji_map = {
            'مقبلات': '🥗',
            'شوربات': '🍲',
            'أطباق رئيسية': '🍽️',
            'أطباق عمانية': '🇴🇲',
            'مشروبات': '🥤',
            'حلويات': '🍰'
        }
        return emoji_map.get(category, '🍽️')
    
    async def _search_menu_items(self, query: str) -> List[Dict]:
        """
        البحث عن الأصناف في القائمة باستخدام Hybrid Search مع Fallback

        استراتيجية البحث الهجين:
        1. تطبيع النص وتصحيح الأخطاء
        2. البحث الموحد (Exact → Partial → Semantic)
        3. Fallback إلى FAISS مع مرونة أعلى
        4. استخراج الصنف من النص المتقطع

        Args:
            query: استعلام البحث

        Returns:
            قائمة الأصناف المطابقة
        """
        try:
            from app.services.menu_search_service import MenuSearchService
            from app.services.faiss_menu_search import faiss_menu_search
            from app.utils.text_helpers import clean_user_query, normalize_order_text
            from app.utils.order_intent_classifier import order_intent_classifier

            # 1. تطبيع النص
            normalized_query = normalize_order_text(query)
            logger.info(f"🔄 Normalized: '{query}' → '{normalized_query}'")

            # 2. تنظيف الاستعلام واستخراج المصطلح الأساسي
            core_term = clean_user_query(normalized_query)
            logger.info(f"🔍 Core term: '{core_term}'")

            # 3. البحث الموحد (المحاولة الأولى)
            search_service = MenuSearchService()
            result = search_service.search_menu(core_term, limit=5)

            if result['results']:
                logger.info(
                    f"✅ Unified search found {result['total_found']} matches "
                    f"(Exact: {result['exact_count']}, "
                    f"Partial: {result['partial_count']}, "
                    f"Semantic: {result['semantic_count']})"
                )
                return result['results']

            # 4. Fallback: FAISS مع مرونة أعلى
            logger.info(f"🔄 Fallback to FAISS with higher flexibility...")
            faiss_results = await faiss_menu_search.search(
                query=core_term,
                top_k=5,
                min_score=0.30,  # مرونة أعلى (كان 0.35)
                keyword_filter=False,  # بدون فلتر للمرونة
                exact_match_boost=True
            )

            if faiss_results:
                logger.info(f"✅ FAISS fallback found {len(faiss_results)} matches")
                # تحويل نتائج FAISS إلى التنسيق المطلوب
                items = [item for item, score in faiss_results]
                return items

            # 5. Fallback النهائي: استخراج الصنف من النص المتقطع
            logger.info(f"🔄 Final fallback: extracting item from fragmented text...")
            extracted = order_intent_classifier.extract_menu_item_from_text(query)

            if extracted and extracted['confidence'] >= 0.5:
                item_name = extracted['item_name']
                logger.info(f"✅ Extracted item: '{item_name}' (confidence: {extracted['confidence']:.2f})")

                # البحث عن الصنف المستخرج
                final_result = search_service.search_menu(item_name, limit=5)
                if final_result['results']:
                    return final_result['results']

            # لم يتم العثور على نتائج
            logger.info(f"ℹ️ No matches found for '{query}' after all fallbacks")
            return []

        except Exception as e:
            logger.error(f"❌ Error searching menu items: {str(e)}")
            import traceback
            logger.error(traceback.format_exc())
            return []
    
    def _format_menu_items(self, items: List[Dict]) -> str:
        """تنسيق الأصناف للـ Prompt"""
        
        if not items:
            return "\n## القائمة:\nلم يتم العثور على أصناف مطابقة."
        
        menu_text = "\n## الأصناف المتاحة:\n"
        for item in items:
            name_ar = item.get('name_ar')
            name_en = item.get('name_en')
            price = item.get('price_omr')
            code = item.get('code')
            
            menu_text += f"- {name_ar} / {name_en} - {price:.3f} ر.ع [{code}]\n"
        
        return menu_text
    
    async def _process_ai_response(
        self,
        ai_reply: str,
        matching_items: List[Dict],
        context: Dict
    ) -> Dict:
        """معالجة رد الـ AI"""

        # محاولة استخراج JSON من الرد
        import json
        import re

        # البحث عن JSON في الرد
        json_match = re.search(r'\{.*\}', ai_reply, re.DOTALL)

        if json_match:
            try:
                response_data = json.loads(json_match.group())
                action = response_data.get('action')
                reply = response_data.get('reply')

                # ✅ تأكد من وجود reply في JSON
                if not reply:
                    logger.warning(f"⚠️ No 'reply' field in JSON response. Using default message.")
                    lang = self.get_language(context)
                    reply = "عذراً، لم أفهم طلبك. هل يمكنك إعادة صياغته؟" if lang == 'ar' else "Sorry, I didn't understand. Can you rephrase?"

                if action == 'add_item':
                    # إضافة الصنف
                    item_code = response_data.get('item_code')

                    if item_code:
                        # لدينا item_code محدد
                        item = next((i for i in matching_items if i['code'] == item_code), None)

                        if item:
                            return self.format_response(
                                intent="New Order - Items",
                                reply=reply,
                                action="add_item",
                                data={
                                    "code": item['code'],
                                    "name": item.get('name_ar', item.get('name_en', 'Unknown')),
                                    "name_ar": item.get('name_ar'),
                                    "name_en": item.get('name_en'),
                                    "quantity": 1,
                                    "price": float(item['price_omr'])
                                }
                            )
                    elif matching_items and len(matching_items) == 1:
                        # لا يوجد item_code لكن لدينا صنف واحد فقط - اعرض معلوماته أولاً
                        item = matching_items[0]
                        logger.info(f"📋 Showing single matching item details: {item['code']}")

                        # Don't add automatically - let AI show item details and ask for confirmation
                        # AI should provide reply with item details
                        return self.format_response(
                            intent="Ask Information",
                            reply=reply,
                            action=None,
                            data=None
                        )
                    else:
                        # لا يوجد item_code ولدينا عدة أصناف - اطلب التوضيح
                        logger.warning(f"⚠️ add_item action without item_code and multiple matches. Using reply only.")
                        return self.format_response(
                            intent="Ask Information",
                            reply=reply,
                            action=None,
                            data=None
                        )

                elif action == 'show_options':
                    # عرض الخيارات
                    return self.format_response(
                        intent="Ask Information",
                        reply=reply,
                        action=None,
                        data=None
                    )

                elif action == 'not_found':
                    # لم يتم العثور
                    return self.format_response(
                        intent="Ask Information",
                        reply=reply,
                        action=None,
                        data=None
                    )

                # ✅ إذا كان هناك action غير معروف، استخدم reply فقط
                logger.warning(f"⚠️ Unknown action: {action}. Using reply only.")
                return self.format_response(
                    intent="Ask Information",
                    reply=reply,
                    action=None,
                    data=None
                )

            except json.JSONDecodeError as e:
                logger.warning(f"⚠️ Failed to parse JSON from AI response: {str(e)}")

        # ✅ إذا لم نستطع استخراج JSON، حاول استخراج النص فقط
        # إزالة أي JSON من الرد
        clean_reply = re.sub(r'\{.*\}', '', ai_reply, flags=re.DOTALL).strip()

        if clean_reply:
            return self.format_response(
                intent="Ask Information",
                reply=clean_reply,
                action=None,
                data=None
            )

        # ✅ إذا لم يكن هناك نص، رد برسالة افتراضية
        lang = self.get_language(context)
        default_reply = "عذراً، لم أفهم طلبك. هل يمكنك إعادة صياغته؟" if lang == 'ar' else "Sorry, I didn't understand. Can you rephrase?"

        return self.format_response(
            intent="Ask Information",
            reply=default_reply,
            action=None,
            data=None
        )
    
    def _timeout_response(self, context: Dict) -> Dict:
        """رد عند Timeout"""
        lang = self.get_language(context)
        
        if lang == 'ar':
            reply = "عذراً، حدث خطأ مؤقت. من فضلك أعد المحاولة."
        else:
            reply = "Sorry, a temporary error occurred. Please try again."
        
        return self.format_response(
            intent="Ask Information",
            reply=reply,
            action=None,
            data=None
        )
    
    async def _clear_selection_state(self, context: Dict):
        """
        Clear old selection state from session
        مسح حالة الاختيار القديمة من الجلسة
        """
        try:
            session = context.get('session', {})
            session_id = session.get('id')

            # 🆕 تحديث السياق المحلي أولاً
            clear_data = {
                'waiting_for_item_choice': False,
                'waiting_for_quantity': False,
                'waiting_for_more_items': False,
                'pending_item_choices': [],
                'pending_items_full': {},
                'selected_item_code': None,
                'selected_item_cache': {}
            }

            # تحديث السياق المحلي
            session.update(clear_data)
            logger.info(f"🧹 Updated local context with cleared state")

            # تحديث قاعدة البيانات
            if session_id:
                from app.services.session_manager import SessionManager
                from app.database.supabase_client import get_supabase_client

                session_manager = SessionManager(get_supabase_client())
                await session_manager.update_session(session_id, clear_data)
                logger.info(f"🧹 Cleared old selection state in database for session {session_id}")
        except Exception as e:
            logger.warning(f"⚠️ Could not clear selection state: {e}")

    def _error_response(self, context: Dict) -> Dict:
        """رد عند خطأ"""
        lang = self.get_language(context)

        if lang == 'ar':
            reply = RTL + "عذراً، حدث خطأ. من فضلك حاول مرة أخرى."  # 🆕 إضافة RTL
        else:
            reply = "Sorry, an error occurred. Please try again."

        return self.format_response(
            intent="Ask Information",
            reply=reply,
            action=None,
            data=None
        )

    async def _check_repeat_order(
        self,
        user_message: str,
        context: Dict
    ) -> Optional[Dict]:
        """
        فحص طلب التكرار (Repeat Order Detection)

        يكشف عبارات مثل:
        - "نفس الشيء"
        - "مرة أخرى"
        - "نفسه"
        - "كمان واحد"
        - "زيادة واحد"

        Args:
            user_message: رسالة المستخدم
            context: سياق المحادثة

        Returns:
            رد الطلب أو None
        """
        try:
            message_lower = user_message.lower().strip()

            # أنماط طلب التكرار
            repeat_patterns = [
                'نفس الشيء',
                'نفس الشي',
                'نفسه',
                'مرة أخرى',
                'مره اخرى',
                'كمان واحد',
                'كمان وحده',
                'زيادة واحد',
                'زياده وحده',
                'نفس الطلب',
                'same thing',
                'same order',
                'one more',
                'another one',
            ]

            # فحص إذا كانت الرسالة تحتوي على أي من الأنماط
            is_repeat_request = any(pattern in message_lower for pattern in repeat_patterns)

            if not is_repeat_request:
                return None

            logger.info("🔁 Repeat order pattern detected")

            # محاولة الحصول على آخر صنف من السياق
            session = context.get('session', {})

            # 1. محاولة الحصول من last_item_name
            last_item_name = session.get('last_item_name')

            if not last_item_name:
                # 2. محاولة الحصول من context_manager
                chat_id = session.get('chat_id')
                if chat_id:
                    from app.services.context_manager import ContextManager
                    from app.database.supabase_client import get_supabase_client

                    context_mgr = ContextManager(get_supabase_client())
                    last_item_name = context_mgr.get_last_item(chat_id)

            if not last_item_name:
                # 3. محاولة الحصول من آخر صنف في الطلب
                items = session.get('items', [])
                if items:
                    last_item_name = items[-1].get('name')

            if not last_item_name:
                logger.warning("⚠️ Repeat order requested but no last item found")

                lang = self.get_language(context)
                if lang == 'ar':
                    reply = "عذراً، لم أتمكن من معرفة الصنف الذي تريد تكراره. من فضلك اذكر اسم الصنف."
                else:
                    reply = "Sorry, I couldn't determine which item you want to repeat. Please specify the item name."

                return self.format_response(
                    intent="Ask Information",
                    reply=reply,
                    action=None,
                    data=None
                )

            logger.info(f"🔁 Repeating last item: '{last_item_name}'")

            # البحث عن الصنف
            matching_items = await self._search_menu_items(last_item_name)

            if not matching_items:
                logger.warning(f"⚠️ Could not find item '{last_item_name}' for repeat order")

                lang = self.get_language(context)
                if lang == 'ar':
                    reply = f"عذراً، لم أتمكن من إيجاد '{last_item_name}'. من فضلك اذكر اسم الصنف."
                else:
                    reply = f"Sorry, I couldn't find '{last_item_name}'. Please specify the item name."

                return self.format_response(
                    intent="Ask Information",
                    reply=reply,
                    action=None,
                    data=None
                )

            # إذا كان هناك صنف واحد فقط، أضفه مباشرة
            if len(matching_items) == 1:
                result = self._handle_single_item(matching_items[0], context)
                # التحقق إذا كانت النتيجة coroutine
                if asyncio.iscoroutine(result):
                    return await result
                return result

            # إذا كان هناك عدة أصناف، اعرضها للاختيار
            result = self._handle_multiple_items(matching_items, context)
            # التحقق إذا كانت النتيجة coroutine
            if asyncio.iscoroutine(result):
                return await result
            return result

        except Exception as e:
            logger.error(f"❌ Error checking repeat order: {str(e)}")
            return None


# Singleton instance
order_food_handler = OrderFoodHandler()

