"""
FAQ Info Handler
معالج الأسئلة العامة (الفروع، المواقع، أوقات العمل)
"""

from typing import Dict, Any, List, Optional
from app.handlers.base_handler import BaseIntentHandler
from app.intents.intent_types import IntentType
import logging
import re

logger = logging.getLogger(__name__)


class FaqInfoHandler(BaseIntentHandler):
    """معالج الأسئلة العامة عن المطعم"""

    @property
    def intent_type(self) -> IntentType:
        return IntentType.FAQ_INFO

    @property
    def prompt_template(self) -> str:
        """قالب الـ Prompt (غير مستخدم لأن الرد ديناميكي حسب السؤال)"""
        return ""
    
    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """
        معالجة الأسئلة العامة
        
        Args:
            user_message: رسالة المستخدم
            context: سياق المحادثة
            conversation_history: تاريخ المحادثة
            
        Returns:
            Dict: الرد على المستخدم
        """
        logger.info(f"❓ FAQ request: {user_message}")
        
        # تحديد نوع السؤال
        message_lower = user_message.lower()
        
        # أسئلة عن الفروع والمواقع
        if any(word in message_lower for word in ['فرع', 'موقع', 'وين', 'فين', 'اين', 'branch', 'location', 'where']):
            response = self._get_branch_info(message_lower)
        
        # أسئلة عن أوقات العمل
        elif any(word in message_lower for word in ['اوقات', 'متى', 'تفتحون', 'hours', 'opening', 'open']):
            response = self._get_opening_hours()
        
        # أسئلة عن التوصيل
        elif any(word in message_lower for word in ['توصيل', 'رسوم', 'deliver', 'delivery', 'fee']):
            response = self._get_delivery_info(message_lower)
        
        # سؤال عام
        else:
            response = self._get_general_info()
        
        return {
            'intent': self.intent_type.value,
            'response': response,
            'action': None,
            'data': {}
        }
    
    def _get_branch_info(self, message: str) -> str:
        """معلومات عن الفروع"""
        # فحص إذا كان يسأل عن مدينة معينة
        cities = {
            'نزوى': 'nizwa',
            'صلالة': 'salalah',
            'صحار': 'sohar',
            'البريمي': 'buraimi',
            'الرستاق': 'rustaq'
        }
        
        for city_ar, city_en in cities.items():
            if city_ar in message or city_en in message:
                return (
                    f"🚧 **فرع {city_ar}**\n\n"
                    f"نعتذر، لا يوجد لدينا فرع في {city_ar} حالياً.\n"
                    "لكن نخطط لافتتاح فروع جديدة قريباً بإذن الله! 🌟\n\n"
                    "📍 **الفرع الرئيسي:**\n"
                    "مسقط - الخوير\n"
                    "بالقرب من مركز القرم التجاري\n\n"
                    "📞 للاستفسار: +968 9000 1234"
                )
        
        # رد عام عن الفروع
        return (
            "📍 **فروعنا**\n\n"
            "🏠 **الفرع الرئيسي:**\n"
            "مسقط - الخوير\n"
            "بالقرب من مركز القرم التجاري\n"
            "📞 +968 9000 1234\n\n"
            "🚧 **قريباً:**\n"
            "• صلالة\n"
            "• نزوى\n"
            "• صحار\n\n"
            "نتطلع لخدمتكم في جميع أنحاء السلطنة! 🇴🇲"
        )
    
    def _get_opening_hours(self) -> str:
        """معلومات عن أوقات العمل"""
        return (
            "🕐 **أوقات العمل**\n\n"
            "📅 **السبت - الخميس:**\n"
            "8:00 صباحاً - 11:00 مساءً\n\n"
            "📅 **الجمعة:**\n"
            "2:00 ظهراً - 11:00 مساءً\n\n"
            "🍽️ **وجبة الإفطار:** 8:00 ص - 11:00 ص\n"
            "🍛 **وجبة الغداء:** 12:00 م - 4:00 م\n"
            "🌙 **وجبة العشاء:** 6:00 م - 11:00 م\n\n"
            "نستقبلكم دائماً بكل سرور! 🌟"
        )
    
    def _get_delivery_info(self, message: str) -> str:
        """معلومات عن التوصيل"""
        # فحص إذا كان يسأل عن منطقة معينة
        if any(word in message for word in ['كم', 'رسوم', 'fee', 'cost']):
            return (
                "🚗 **خدمة التوصيل**\n\n"
                "✅ نوفر خدمة التوصيل لجميع مناطق مسقط\n\n"
                "💰 **رسوم التوصيل:**\n"
                "• داخل الخوير: مجاناً 🎉\n"
                "• مسقط (حتى 10 كم): 500 بيسة\n"
                "• مسقط (10-20 كم): 1.000 ريال\n"
                "• خارج مسقط: حسب المسافة\n\n"
                "⏱️ **وقت التوصيل:**\n"
                "30-45 دقيقة في المتوسط\n\n"
                "📦 **توصيل مجاني** للطلبات فوق 10 ريال! 🎁"
            )
        
        return (
            "🚗 **خدمة التوصيل**\n\n"
            "✅ نوفر خدمة التوصيل لجميع مناطق مسقط\n"
            "⏱️ وقت التوصيل: 30-45 دقيقة\n"
            "💰 رسوم التوصيل تبدأ من 500 بيسة\n"
            "🎁 توصيل مجاني للطلبات فوق 10 ريال\n\n"
            "📞 للاستفسار: +968 9000 1234"
        )
    
    def _get_general_info(self) -> str:
        """معلومات عامة"""
        return (
            "ℹ️ **معلومات عامة**\n\n"
            "🏠 **الموقع:**\n"
            "مسقط - الخوير\n\n"
            "🕐 **أوقات العمل:**\n"
            "السبت - الخميس: 8:00 ص - 11:00 م\n"
            "الجمعة: 2:00 م - 11:00 م\n\n"
            "🚗 **التوصيل:**\n"
            "متوفر لجميع مناطق مسقط\n\n"
            "📞 **للتواصل:**\n"
            "+968 9000 1234\n"
            "support@tasteoman.com\n\n"
            "كيف يمكنني مساعدتك؟ 😊"
        )

