"""
Edit Item Handler
معالج تعديل الأصناف في السلة
"""

from typing import Dict, Any, List, Optional
from app.handlers.base_handler import BaseIntentHandler
from app.intents.definitions import IntentType  # ✅ استخدام definitions بدلاً من intent_types
from app.utils.text_helpers import extract_quantity_and_item
from app.services.cart_service import cart_service
from app.utils.cart_formatter import cart_formatter
import logging
import re

logger = logging.getLogger(__name__)


class EditItemHandler(BaseIntentHandler):
    """معالج تعديل كمية صنف في السلة"""

    @property
    def intent_type(self) -> IntentType:
        return IntentType.MODIFY_ORDER  # ✅ استخدام MODIFY_ORDER بدلاً من EDIT_ITEM

    @property
    def prompt_template(self) -> str:
        return ""  # Not used for this handler
    
    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """
        معالجة طلب تعديل صنف

        أمثلة:
        - "خلي الفتوش 1"
        - "غير الهريس 3"
        - "عدل الكبسة 2"
        - "خليها بس واحدة" (يستخدم آخر صنف)
        """
        logger.info(f"🔧 Handling edit item: '{user_message}'")

        # الحصول على الجلسة
        session = context.get('session', {})
        session_id = session.get('id')
        lang = session.get('lang', 'ar')

        # الحصول على الأصناف من cart_service
        items = cart_service.get_cart_items(session_id)

        # التحقق من وجود أصناف في السلة
        if not items:
            return self.format_response(
                intent="Ask Information",
                reply="🤔 السلة فارغة حالياً. لا يوجد أصناف للتعديل.\n\nجرب إضافة صنف أولاً!",
                action=None
            )
        
        # استخراج اسم الصنف والكمية الجديدة
        quantity, item_name = extract_quantity_and_item(user_message)
        
        logger.info(f"📊 Extracted: quantity={quantity}, item='{item_name}'")
        
        # إذا لم يتم ذكر اسم الصنف، استخدم آخر صنف تم إضافته
        if not item_name:
            item_name = session.get('last_ordered_item')
            logger.info(f"💾 Using last ordered item: '{item_name}'")
        
        # إذا لم يتم العثور على اسم الصنف
        if not item_name:
            return self.format_response(
                intent="Ask Information",
                reply="🤔 أي صنف ترغب في تعديله؟\n\nمثال: خلي الفتوش 1",
                action=None
            )
        
        # البحث عن الصنف في السلة
        item_found = None
        item_code = None

        # تنظيف اسم الصنف من "ال" التعريف
        clean_item_name = item_name.replace('ال', '').replace('الـ', '').strip().lower()

        logger.info(f"🔍 Searching for: '{item_name}' → cleaned: '{clean_item_name}'")

        for cart_item in items:
            cart_name_ar = cart_item.get('item_name_ar', '').lower()
            cart_name_en = cart_item.get('item_name_en', '').lower()
            cart_name = cart_item.get('item_name', '').lower()

            # تنظيف أسماء السلة أيضاً
            clean_cart_ar = cart_name_ar.replace('ال', '').replace('الـ', '').strip()
            clean_cart_en = cart_name_en.replace('ال', '').replace('الـ', '').strip()
            clean_cart = cart_name.replace('ال', '').replace('الـ', '').strip()

            # البحث بالاسم الأصلي أو المنظف
            if (clean_item_name in clean_cart_ar or
                clean_item_name in clean_cart_en or
                clean_item_name in clean_cart or
                item_name.lower() in cart_name_ar or
                item_name.lower() in cart_name_en or
                item_name.lower() in cart_name):
                item_found = cart_item
                item_code = cart_item.get('item_code')
                logger.info(f"✅ Found match: '{cart_item.get('item_name_ar', cart_item.get('item_name'))}'")
                break

        # إذا لم يتم العثور على الصنف في السلة
        if not item_found:
            # عرض قائمة الأصناف الموجودة
            items_list = "\n".join([f"• {item.get('item_name_ar', item.get('item_name', ''))} (×{item.get('qty', 1)})"
                                   for item in items])

            return self.format_response(
                intent="Ask Information",
                reply=f"❌ لم أجد '{item_name}' في سلتك.\n\n📦 الأصناف الموجودة:\n{items_list}\n\nجرب تعديل أحد هذه الأصناف.",
                action=None
            )
        
        # إذا كانت الكمية الجديدة 0، احذف الصنف
        if quantity == 0:
            success = cart_service.remove_item(session_id, item_code)

            if not success:
                return self.format_response(
                    intent="Error",
                    reply="❌ حدث خطأ أثناء حذف الصنف. يرجى المحاولة مرة أخرى.",
                    action=None
                )

            # الحصول على ملخص السلة الجديد
            cart_summary_data = cart_service.get_cart_summary(session_id, lang)
            totals = {
                'subtotal': cart_summary_data['subtotal'],
                'delivery_fee': cart_summary_data['delivery_fee'],
                'discount': cart_summary_data['discount'],
                'total': cart_summary_data['total']
            }

            item_name_display = item_found.get('item_name_ar', item_found.get('item_name', ''))
            reply = f"🗑️ تم حذف {item_name_display} من سلتك.\n\n"

            if cart_summary_data['items']:
                reply += f"💰 المجموع الجديد: {totals['total']:.3f} ر.ع"
            else:
                reply += "السلة أصبحت فارغة."

            return self.format_response(
                intent="Confirm",
                reply=reply,
                action=None
            )

        # تحديث الكمية
        old_quantity = item_found.get('qty', 1)
        updated_item = cart_service.update_item_quantity(session_id, item_code, quantity)

        if not updated_item:
            return self.format_response(
                intent="Error",
                reply="❌ حدث خطأ أثناء تعديل الكمية. يرجى المحاولة مرة أخرى.",
                action=None
            )

        # الحصول على ملخص السلة الجديد
        cart_summary_data = cart_service.get_cart_summary(session_id, lang)
        totals = {
            'subtotal': cart_summary_data['subtotal'],
            'delivery_fee': cart_summary_data['delivery_fee'],
            'discount': cart_summary_data['discount'],
            'total': cart_summary_data['total']
        }

        # تنسيق الرد
        item_name_display = item_found.get('item_name_ar', item_found.get('item_name', ''))
        price_per_item = item_found.get('unit_price', 0)
        new_total_for_item = price_per_item * quantity

        reply = f"✅ تم تعديل {item_name_display}\n\n"
        reply += f"📊 الكمية: {old_quantity} ← {quantity}\n"
        reply += f"💵 السعر: {new_total_for_item:.3f} ر.ع\n\n"
        reply += f"💰 المجموع الكلي: {totals['total']:.3f} ر.ع"

        return self.format_response(
            intent="Confirm",
            reply=reply,
            action=None
        )

