"""
Confirm Order Handler - معالج تأكيد الطلب
يستخدم Cart Formatter الموحد
"""

from typing import Dict, Optional, List
from app.handlers.base_handler import BaseIntentHandler
from app.intents.definitions import IntentType
from app.utils.cart_formatter import cart_formatter
from app.utils.logger import app_logger as logger
from app.services.cart_service import cart_service
from app.state_machine.order_states import OrderPhase


class ConfirmOrderHandler(BaseIntentHandler):
    """معالج تأكيد الطلب (نعم، موافق، أكمل)"""
    
    @property
    def intent_type(self) -> IntentType:
        return IntentType.CONFIRM_ORDER
    
    @property
    def prompt_template(self) -> str:
        return """أنت مساعد مطعم. المستخدم يريد تأكيد الطلب والمتابعة."""
    
    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """
        معالجة تأكيد الطلب مع التحقق من الحالة والصنف المعلق

        يتحقق من:
        1. وجود أصناف في السلة
        2. حالة الطلب الحالية (state)
        3. وجود صنف معلق (pending_item)
        4. صحة الانتقال بين المراحل
        """
        try:
            logger.info(f"✅ Handling confirm order")

            # الحصول على المرحلة والحالة الحالية
            stage = context.get('stage', 'items')
            state = context.get('state')  # الحالة الديناميكية
            session = context.get('session', {})
            session_id = session.get('id')
            lang = self.get_language(context)

            # 🆕 حماية: إذا كانت الجلسة مكتملة بالفعل، لا تعيد السؤال
            session_status = session.get('status', 'active')
            if session_status == 'completed':
                logger.warning(f"⚠️ Session already completed, ignoring confirm request")
                return None  # لا ترد بشيء

            # التحقق من وجود صنف معلق
            pending_item = context.get('pending_item')
            if pending_item:
                logger.info(f"⚠️ Pending item detected: {pending_item}")
                # يجب إكمال إضافة الصنف المعلق أولاً
                return self._handle_pending_item_confirmation(pending_item, context, lang)

            # 🆕 Get cart items from cart_service (reads from draft_cart_items table)
            try:
                cart_summary = cart_service.get_cart_summary(session_id, lang)
                cart_items = cart_summary.get('items', [])
            except Exception as e:
                logger.error(f"❌ Error getting cart: {e}")
                cart_items = []

            # التحقق من وجود أصناف في السلة
            if not cart_items:
                return self._empty_cart_response(lang)

            # التحقق من الحالة الديناميكية
            if state == 'awaiting_confirmation':
                logger.info(f"✅ State is awaiting_confirmation, proceeding with confirmation")
                # مسح الحالة بعد التأكيد
                context['state'] = None
                if 'pending_item' in context:
                    del context['pending_item']

            # حسب المرحلة
            if stage == 'items':
                # التحقق: هل لديه عنوان مسبق؟
                has_address = context.get('address') or context.get('latitude')

                if has_address:
                    # لديه عنوان → الانتقال للدفع مباشرة
                    logger.info("✅ User has address, proceeding to payment")
                    return self._proceed_to_payment(lang)
                else:
                    # لا يوجد عنوان → اطلب العنوان
                    logger.info("📍 No address found, requesting address")
                    return self._proceed_to_address(lang)

            elif stage == 'address':
                # الانتقال لمرحلة الدفع
                return self._proceed_to_payment(lang)

            elif stage == 'payment':
                # إنشاء الطلب
                return self._create_order(lang)

            else:
                # افتراضي: الانتقال للعنوان
                return self._proceed_to_address(lang)

        except Exception as e:
            logger.error(f"❌ Error in confirm order handler: {str(e)}")
            return self._error_response(context)
    
    def _proceed_to_address(self, lang: str) -> Dict:
        """الانتقال لمرحلة العنوان"""
        
        if lang == 'ar':
            reply = "ممتاز! 📍\n\nمن فضلك أرسل عنوان التوصيل.\n\nيمكنك إرسال:\n• نص (مثل: مسقط، الخوير، شارع السلطان قابوس)\n• موقع GPS"
        else:
            reply = "Great! 📍\n\nPlease send your delivery address.\n\nYou can send:\n• Text (e.g., Muscat, Al Khuwair, Sultan Qaboos Street)\n• GPS location"
        
        return self.format_response(
            intent="Confirm Order",
            reply=reply,
            action="update_session",
            data={
                "stage": "address",
                "order_phase": OrderPhase.ADDRESS_INFO.value,  # 🔧 تحديث المرحلة
                "waiting_for_more_items": False  # 🆕 Clear flag when proceeding to address
            }
        )
    
    def _proceed_to_payment(self, lang: str) -> Dict:
        """الانتقال لمرحلة الدفع"""

        if lang == 'ar':
            reply = "شكراً! 💳\n\nاختر طريقة الدفع:\n1. الدفع عند الاستلام (نقداً)\n2. بطاقة ائتمان"
        else:
            reply = "Thank you! 💳\n\nChoose payment method:\n1. Cash on delivery\n2. Credit card"

        return self.format_response(
            intent="Confirm Order",
            reply=reply,
            action="update_session",
            data={
                "stage": "payment",
                "waiting_for_more_items": False  # 🆕 Clear flag when proceeding to payment
            }
        )
    
    def _create_order(self, lang: str) -> Dict:
        """إنشاء الطلب"""
        
        if lang == 'ar':
            reply = "جاري إنشاء طلبك... ⏳"
        else:
            reply = "Creating your order... ⏳"
        
        return self.format_response(
            intent="Confirm Order",
            reply=reply,
            action="create_order",
            data={}
        )
    
    def _empty_cart_response(self, lang: str) -> Dict:
        """رد عندما تكون السلة فارغة"""

        reply = cart_formatter.format_empty_cart_message(lang)

        return self.format_response(
            intent="Ask Information",
            reply=reply,
            action=None,
            data=None
        )
    
    def _error_response(self, context: Dict) -> Dict:
        """رد عند خطأ"""
        lang = self.get_language(context)
        
        if lang == 'ar':
            reply = "عذراً، حدث خطأ. من فضلك حاول مرة أخرى."
        else:
            reply = "Sorry, an error occurred. Please try again."
        
        return self.format_response(
            intent="Ask Information",
            reply=reply,
            action=None,
            data=None
        )
    def _handle_pending_item_confirmation(
        self,
        pending_item: str,
        context: Dict,
        lang: str
    ) -> Dict:
        """
        معالجة تأكيد صنف معلق

        Args:
            pending_item: اسم الصنف المعلق
            context: السياق الحالي
            lang: اللغة

        Returns:
            استجابة التأكيد
        """
        logger.info(f"🔄 Handling pending item confirmation: {pending_item}")

        # رسالة تأكيد
        if lang == 'ar':
            reply = f"⚠️ لديك صنف معلق: {pending_item}\n\n"
            reply += "هل تريد إضافته إلى السلة؟\n\n"
            reply += "✅ نعم - إضافة الصنف\n"
            reply += "❌ لا - إلغاء"
        else:
            reply = f"⚠️ You have a pending item: {pending_item}\n\n"
            reply += "Would you like to add it to your cart?\n\n"
            reply += "✅ Yes - Add item\n"
            reply += "❌ No - Cancel"

        return self.format_response(
            intent="Ask Information",
            reply=reply,
            action=None,
            data=None
        )


# Singleton instance
confirm_order_handler = ConfirmOrderHandler()

