"""
Clear Cart Handler - معالج حذف/مسح السلة
"""

from typing import Dict, Optional, List
from app.handlers.base_handler import BaseIntentHandler
from app.intents.definitions import IntentType
from app.utils.logger import app_logger as logger
from app.services.cart_service import cart_service
from app.utils.question_context_manager import question_context_manager, QuestionType


class ClearCartHandler(BaseIntentHandler):
    """معالج حذف/مسح السلة"""
    
    @property
    def intent_type(self) -> IntentType:
        return IntentType.CLEAR_CART
    
    @property
    def prompt_template(self) -> str:
        return """أنت مساعد مطعم ذكي. مهمتك: حذف/مسح السلة.

## القواعد:
1. اسأل العميل للتأكيد قبل حذف السلة
2. إذا أكد، احذف جميع الأصناف من السلة
3. أرسل رسالة تأكيد بعد الحذف

## التنسيق:
- رد بالعربية إذا كان العميل يتحدث بالعربية
- رد بالإنجليزية إذا كان العميل يتحدث بالإنجليزية
- كن واضحاً ومهذباً
"""
    
    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """معالجة حذف السلة"""

        try:
            logger.info(f"🗑️ Handling clear cart request")

            # الحصول على البيانات
            session = context.get('session', {})
            session_id = session.get('id')
            lang = self.get_language(context)

            # التحقق من وجود سلة
            try:
                cart_summary = cart_service.get_cart_summary(session_id, lang)
                cart_items = cart_summary.get('items', [])
            except Exception as e:
                logger.error(f"❌ Error getting cart: {e}")
                cart_items = []

            # إذا كانت السلة فارغة بالفعل
            if not cart_items or len(cart_items) == 0:
                if lang == 'ar':
                    reply = "السلة فارغة بالفعل! 🛍️\n\nهل تريد البدء بطلب جديد؟"
                else:
                    reply = "The cart is already empty! 🛍️\n\nWould you like to start a new order?"
                
                return self.format_response(
                    intent="Clear Cart",
                    reply=reply,
                    action=None,
                    data=None
                )

            # طلب التأكيد من العميل
            logger.info(f"❓ Asking for confirmation to clear cart")
            
            # إضافة السؤال للسياق
            question_context_manager.set_question(session, QuestionType.CLEAR_CART)
            
            # الحصول على رسالة السؤال
            confirmation_message = question_context_manager.get_question_message(
                QuestionType.CLEAR_CART,
                lang
            )

            return self.format_response(
                intent="Clear Cart",
                reply=confirmation_message,
                action="update_session",
                data={
                    "awaiting_confirmation": True,
                    "last_question": {
                        "type": QuestionType.CLEAR_CART.value,
                        "asked_at": None  # سيتم تعيينه في webhook
                    }
                }
            )

        except Exception as e:
            logger.error(f"❌ Error in clear cart handler: {str(e)}")
            return self._error_response(context)
    
    
    def _error_response(self, context: Dict) -> Dict:
        """رد عند خطأ"""
        lang = self.get_language(context)
        
        if lang == 'ar':
            reply = "عذراً، حدث خطأ في حذف السلة. من فضلك حاول مرة أخرى."
        else:
            reply = "Sorry, an error occurred while clearing the cart. Please try again."
        
        return self.format_response(
            intent="Clear Cart",
            reply=reply,
            action=None,
            data=None
        )


# Singleton instance
clear_cart_handler = ClearCartHandler()

