"""
Base Handler - الفئة الأساسية لجميع معالجات النوايا
"""

from abc import ABC, abstractmethod
from typing import Dict, Optional, List
from openai import AsyncOpenAI
import asyncio
from app.config import settings
from app.utils.logger import app_logger as logger
from app.intents.definitions import IntentType


class BaseIntentHandler(ABC):
    """الفئة الأساسية لمعالج النية"""
    
    def __init__(self):
        self.client = AsyncOpenAI(
            api_key=settings.OPENAI_API_KEY,
            timeout=30.0
        )
        self.model = "gpt-4o-mini"
    
    @property
    @abstractmethod
    def intent_type(self) -> IntentType:
        """نوع النية التي يعالجها هذا Handler"""
        pass
    
    @property
    @abstractmethod
    def prompt_template(self) -> str:
        """قالب الـ Prompt الخاص بهذه النية"""
        pass
    
    @abstractmethod
    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """
        معالجة النية
        
        Args:
            user_message: رسالة المستخدم
            context: السياق (السلة، المرحلة، إلخ)
            conversation_history: تاريخ المحادثة
            
        Returns:
            Dict: الرد (intent, reply, action, data)
        """
        pass
    
    async def get_ai_response(
        self,
        user_message: str,
        system_prompt: str,
        conversation_history: Optional[List[Dict]] = None,
        timeout: float = 20.0
    ) -> str:
        """
        الحصول على رد من OpenAI
        
        Args:
            user_message: رسالة المستخدم
            system_prompt: System prompt
            conversation_history: تاريخ المحادثة
            timeout: Timeout بالثواني
            
        Returns:
            str: رد الـ AI
        """
        try:
            messages = [{"role": "system", "content": system_prompt}]
            
            # إضافة تاريخ المحادثة (آخر 3 رسائل فقط)
            if conversation_history:
                for msg in conversation_history[-3:]:
                    messages.append({
                        "role": msg.get("role"),
                        "content": msg.get("content")
                    })
            
            # إضافة رسالة المستخدم
            messages.append({"role": "user", "content": user_message})
            
            # استدعاء OpenAI
            start_time = asyncio.get_event_loop().time()
            
            response = await asyncio.wait_for(
                self.client.chat.completions.create(
                    model=self.model,
                    messages=messages,
                    temperature=0.3,
                    max_tokens=500
                ),
                timeout=timeout
            )
            
            end_time = asyncio.get_event_loop().time()
            duration = end_time - start_time
            
            reply = response.choices[0].message.content.strip()
            
            logger.info(f"✅ AI response received ({duration:.2f}s, {response.usage.total_tokens} tokens)")
            
            return reply
            
        except asyncio.TimeoutError:
            logger.error(f"❌ AI response timeout after {timeout}s")
            raise TimeoutError(f"AI response timeout after {timeout}s")
            
        except Exception as e:
            logger.error(f"❌ Error getting AI response: {str(e)}")
            raise
    
    def build_prompt(
        self,
        context: Dict,
        additional_info: Optional[str] = None
    ) -> str:
        """
        بناء الـ Prompt من القالب
        
        Args:
            context: السياق
            additional_info: معلومات إضافية
            
        Returns:
            str: الـ Prompt الكامل
        """
        prompt = self.prompt_template
        
        # إضافة معلومات السياق
        if context:
            stage = context.get('stage', 'items')
            lang = context.get('lang', 'ar')
            cart_items = context.get('items', [])
            
            prompt += f"\n\n## السياق الحالي:\n"
            prompt += f"- المرحلة: {stage}\n"
            prompt += f"- اللغة: {lang}\n"
            prompt += f"- عدد الأصناف في السلة: {len(cart_items)}\n"
            
            if cart_items:
                prompt += f"\n### السلة الحالية:\n"
                for item in cart_items:
                    prompt += f"- {item.get('name')} × {item.get('quantity')} = {item.get('price')} ر.ع\n"
        
        # إضافة معلومات إضافية
        if additional_info:
            prompt += f"\n\n{additional_info}"
        
        return prompt
    
    def format_response(
        self,
        intent: str,
        reply: str,
        action: Optional[str] = None,
        data: Optional[Dict] = None
    ) -> Dict:
        """
        تنسيق الرد
        
        Args:
            intent: النية
            reply: الرد النصي
            action: الإجراء (اختياري)
            data: البيانات (اختياري)
            
        Returns:
            Dict: الرد المنسق
        """
        return {
            "intent": intent,
            "reply": reply,
            "action": action,
            "data": data or {}
        }
    
    def get_language(self, context: Dict) -> str:
        """الحصول على لغة المحادثة"""
        return context.get('lang', 'ar')
    
    def is_arabic(self, text: str) -> bool:
        """التحقق إذا كان النص عربي"""
        return any(ord(c) > 127 for c in text[:50])

