"""
Ask Price Handler - معالج السؤال عن السعر
"""

from typing import Dict, Optional, List
from app.handlers.base_handler import BaseIntentHandler
from app.intents.definitions import IntentType
from app.services.faiss_menu_search import faiss_menu_search
from app.utils.logger import app_logger as logger


class AskPriceHandler(BaseIntentHandler):
    """معالج السؤال عن السعر"""
    
    @property
    def intent_type(self) -> IntentType:
        return IntentType.ASK_PRICE
    
    @property
    def prompt_template(self) -> str:
        return """أنت مساعد مطعم. مهمتك: الإجابة عن سؤال العميل عن السعر.

## القواعد:
- أجب عن السعر بوضوح
- اذكر اسم الصنف والسعر
- اسأل إذا كان يريد إضافته للسلة
- كن مختصراً ومباشراً
"""
    
    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """معالجة السؤال عن السعر"""

        try:
            logger.info(f"💰 Handling ask price: '{user_message}'")

            # 1. استخراج اسم الصنف من السؤال
            item_name = self._extract_item_name(user_message)

            # 💾 2. إذا لم يُذكر اسم صنف، حاول استرجاع آخر صنف من السياق
            if not item_name or len(item_name) < 2:
                item_name = self._get_last_item_from_context(context)
                if item_name:
                    logger.info(f"📖 Using last item from context: '{item_name}'")
                else:
                    logger.warning("⚠️ No item name in query and no context available")

            # 3. البحث عن الصنف باستخدام FAISS
            matching_items = await self._search_item(item_name)

            # 🆕 4. فحص الغموض: إذا كان هناك خيارات متعددة في السياق
            ambiguity_response = self._check_ambiguity(
                matching_items,
                context,
                conversation_history
            )
            if ambiguity_response:
                logger.info("⚠️ Ambiguous reference detected - asking for clarification")
                return ambiguity_response

            # 5. تنسيق الرد
            lang = self.get_language(context)
            reply = self._format_price_response(matching_items, item_name, lang)

            # 🆕 6. إذا كان هناك عدة أصناف، احفظ الحالة للانتظار اختيار المستخدم
            if matching_items and len(matching_items) > 1:
                logger.info(f"💾 Multiple items found ({len(matching_items)}), saving state for user selection")

                # 🆕 تخزين العناصر الكاملة لتجنب الاستعلامات المكررة
                pending_items_full = {item['code']: item for item in matching_items}

                return self.format_response(
                    intent="Price Information",
                    reply=reply,
                    action="update_session",
                    data={
                        "waiting_for_item_choice": True,
                        "pending_item_choices": [item['code'] for item in matching_items],
                        "pending_items_full": pending_items_full,  # 🆕 تخزين العناصر الكاملة
                        "pending_quantity": 1,
                        "waiting_for_quantity": False,
                        "waiting_for_more_items": False,
                        "items": matching_items  # للرجوع إليها لاحقاً
                    }
                )

            # صنف واحد فقط أو لا توجد نتائج
            return self.format_response(
                intent="Price Information",
                reply=reply,
                action=None,
                data={"items": matching_items} if matching_items else None
            )

        except Exception as e:
            logger.error(f"❌ Error in ask price handler: {str(e)}")
            return self._error_response(context)
    
    def _normalize_arabic_text(self, text: str) -> str:
        """
        تطبيع النص العربي:
        - توحيد الهمزات والياءات والتاء المربوطة
        - إزالة التشكيل
        - إزالة "ال" التعريف
        """
        from app.utils.text_helpers import normalize_arabic_spelling
        import re

        # 1. توحيد الاختلافات الإملائية
        text = normalize_arabic_spelling(text)

        # 2. إزالة "ال" التعريف من بداية الكلمات
        # مثال: "اللحم" → "لحم"
        words = text.split()
        normalized_words = []

        for word in words:
            # إذا كانت الكلمة تبدأ بـ "ال" وطولها أكثر من 3 أحرف
            if word.startswith('ال') and len(word) > 3:
                # إزالة "ال" من البداية
                word = word[2:]
            normalized_words.append(word)

        return ' '.join(normalized_words)

    def _extract_item_name(self, message: str) -> str:
        """استخراج اسم الصنف من السؤال"""

        # إزالة كلمات السؤال الشائعة (يجب أن تتطابق مع intent_guard.py)
        price_keywords = [
            # الكلمات الأساسية
            "سعر", "كم", "كام", "ثمن", "بكم", "بكام", "قيمة", "تكلفة", "كلفة",
            # مع الضمائر
            "سعره", "سعرها", "ثمنه", "ثمنها", "قيمته", "قيمتها", "كلفته",
            # الأفعال (مهم جداً!)
            "يكلف", "تكلف", "يطلع", "يصير", "أدفع", "ادفع",
            # اللهجات
            "بجم", "بچم", "حق", "حقه", "شحالة", "قديش", "كديش", "أديش", "بدو",
            # الصفات
            "غالي", "رخيص", "معقول", "أرخص", "أغلى",
            # الكلمات المساعدة
            "المبلغ", "المجموع", "الإجمالي", "الحساب",
            # English
            "price", "cost", "how much", "value", "expensive", "cheap",
            "total", "amount", "pay", "bill",
            # كلمات عامة
            "ما", "what", "what's", "whats", "ال", "الـ", "في", "عندكم", "لديكم"
        ]

        words = message.lower().split()
        item_words = []

        for word in words:
            # تنظيف الكلمة من علامات الترقيم
            clean_word = word.strip("؟?.,!،")

            # تجاهل كلمات السؤال
            if clean_word not in price_keywords:
                item_words.append(clean_word)

        # إرجاع اسم الصنف
        item_name = " ".join(item_words).strip()

        # تطبيع النص العربي (إزالة "ال" التعريف)
        normalized_name = self._normalize_arabic_text(item_name)

        logger.info(f"📝 Extracted item name: '{item_name}' → normalized: '{normalized_name}'")

        return normalized_name

    def _check_ambiguity(
        self,
        matching_items: List[Dict],
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Optional[Dict]:
        """
        فحص الغموض في السؤال

        إذا كان البوت عرض خيارات متعددة في الرسالة السابقة،
        والمستخدم سأل "كم سعره؟" بدون تحديد، نسأله للتوضيح

        Args:
            matching_items: الأصناف المطابقة
            context: سياق المحادثة
            conversation_history: سجل المحادثة

        Returns:
            رد طلب التوضيح أو None
        """
        try:
            # فحص إذا كان هناك خيارات متعددة في السياق
            session = context.get('session', {})
            pending_choices = session.get('pending_item_choices', [])

            # إذا كان هناك خيارات معلقة وأكثر من خيار واحد
            if pending_choices and len(pending_choices) > 1:
                logger.info(f"⚠️ Ambiguous reference: {len(pending_choices)} options pending")

                # بناء رسالة طلب التوضيح
                lang = self.get_language(context)

                if lang == 'ar':
                    reply = "أي نوع تقصد؟\n\n"
                    for i, item in enumerate(pending_choices[:5], 1):
                        name = item.get('name_ar', item.get('name_en', 'Unknown'))
                        price = item.get('price', 0)
                        reply += f"{i}. {name} - {price} OMR\n"
                else:
                    reply = "Which one do you mean?\n\n"
                    for i, item in enumerate(pending_choices[:5], 1):
                        name = item.get('name_en', item.get('name_ar', 'Unknown'))
                        price = item.get('price', 0)
                        reply += f"{i}. {name} - {price} OMR\n"

                return self.format_response(
                    intent="Clarification Request",
                    reply=reply.strip(),
                    action=None,
                    data={"pending_choices": pending_choices}
                )

            # فحص إذا كان البوت عرض خيارات في آخر رسالة
            if conversation_history and len(conversation_history) > 0:
                last_bot_message = None
                for msg in reversed(conversation_history):
                    if msg.get('role') == 'assistant':
                        last_bot_message = msg.get('content', '')
                        break

                if last_bot_message:
                    # فحص إذا كانت الرسالة تحتوي على خيارات مرقمة
                    has_options = any(marker in last_bot_message for marker in ['1.', '2.', '1-', '2-', '١.', '٢.'])

                    # فحص إذا كانت الرسالة تحتوي على "أو" (عرض خيارات)
                    has_or = ' أو ' in last_bot_message or ' or ' in last_bot_message.lower()

                    # فحص إذا كانت الرسالة تحتوي على "و" (عرض خيارات)
                    # مثال: "عندنا هريس دجاج وهريس لحم"
                    has_and = ' و' in last_bot_message and 'عندنا' in last_bot_message

                    if (has_options or has_or or has_and) and len(matching_items) >= 1:
                        logger.info("⚠️ Ambiguous reference: bot presented options in last message")

                        # بناء رسالة طلب التوضيح
                        lang = self.get_language(context)

                        if lang == 'ar':
                            reply = "أي نوع تقصد؟\n\n"
                            for i, item in enumerate(matching_items[:5], 1):
                                name = item.get('name_ar', item.get('name_en', 'Unknown'))
                                price = item.get('price', 0)
                                reply += f"{i}. {name} - {price} OMR\n"
                        else:
                            reply = "Which one do you mean?\n\n"
                            for i, item in enumerate(matching_items[:5], 1):
                                name = item.get('name_en', item.get('name_ar', 'Unknown'))
                                price = item.get('price', 0)
                                reply += f"{i}. {name} - {price} OMR\n"

                        return self.format_response(
                            intent="Clarification Request",
                            reply=reply.strip(),
                            action=None,
                            data={"items": matching_items}
                        )

            return None

        except Exception as e:
            logger.warning(f"⚠️ Error checking ambiguity: {e}")
            return None

    def _get_last_item_from_context(self, context: Dict) -> Optional[str]:
        """
        استرجاع آخر صنف من السياق

        Args:
            context: سياق المحادثة

        Returns:
            اسم الصنف أو None
        """
        try:
            # 1. Try to get from session (last_item_name)
            session = context.get('session', {})
            last_item = session.get('last_item_name')

            if last_item:
                logger.info(f"📖 Found last item in session: '{last_item}'")
                return last_item

            # 2. Try to get from context manager (if available)
            # This will be set by webhook.py
            chat_id = session.get('chat_id')
            if chat_id:
                # Import here to avoid circular dependency
                from app.services.context_manager import ContextManager
                from app.database.supabase_client import get_supabase_client

                context_mgr = ContextManager(get_supabase_client())
                last_item = context_mgr.get_last_item(chat_id)

                if last_item:
                    logger.info(f"📖 Found last item in context manager: '{last_item}'")
                    return last_item

            logger.info("ℹ️ No last item found in context")
            return None

        except Exception as e:
            logger.warning(f"⚠️ Error getting last item from context: {e}")
            return None

    async def _search_item(self, item_name: str) -> List[Dict]:
        """البحث عن الصنف باستخدام FAISS مع fallback للبحث النصي"""

        try:
            if not item_name:
                return []

            # Try 1: Simple text search first (fastest and most accurate for exact matches)
            simple_results = self._simple_text_search(item_name)
            if simple_results:
                logger.info(f"✅ Simple text search found {len(simple_results)} matches")
                return simple_results[:3]  # Return top 3

            # Try 2: FAISS search with keyword filter
            if faiss_menu_search.is_loaded():
                logger.info(f"🔍 Searching with FAISS: '{item_name}'")

                search_results = await faiss_menu_search.search(
                    item_name,
                    top_k=3,
                    min_score=0.2,
                    keyword_filter=True
                )

                # Try 3: If no results, try without keyword filter
                if not search_results:
                    logger.info(f"🔄 Retrying without keyword filter...")
                    search_results = await faiss_menu_search.search(
                        item_name,
                        top_k=3,
                        min_score=0.2,
                        keyword_filter=False
                    )

                if search_results:
                    matching_items = [item for item, score in search_results]
                    logger.info(f"✅ FAISS found {len(matching_items)} matching items")
                    return matching_items
                else:
                    logger.warning(f"⚠️ FAISS found no matches for '{item_name}'")
                    return []
            else:
                logger.warning("⚠️ FAISS not loaded")
                return []

        except Exception as e:
            logger.error(f"❌ Error searching item: {str(e)}")
            return []

    def _simple_text_search(self, query: str) -> List[Dict]:
        """
        بحث نصي بسيط في أسماء الأصناف

        Args:
            query: نص البحث

        Returns:
            قائمة الأصناف المطابقة
        """
        try:
            import json
            import os

            # Load menu from cache
            cache_path = os.path.join(
                os.path.dirname(os.path.dirname(os.path.dirname(__file__))),
                'data', 'cache', 'menu_cache.json'
            )

            if not os.path.exists(cache_path):
                return []

            with open(cache_path, 'r', encoding='utf-8') as f:
                menu_data = json.load(f)

            items = menu_data.get('items', [])
            query_lower = query.lower().strip()
            matches = []

            for item in items:
                if not item.get('active', True):
                    continue

                name_ar = (item.get('name_ar') or '').lower()
                name_en = (item.get('name_en') or '').lower()
                tags = [tag.lower() for tag in (item.get('tags') or [])]

                # Check for matches
                score = 0

                # Exact match in name (highest priority)
                if query_lower == name_ar or query_lower == name_en:
                    score = 100
                # Query is in name (high priority)
                elif query_lower in name_ar or query_lower in name_en:
                    score = 90
                # Query word is in name words
                elif query_lower in name_ar.split() or query_lower in name_en.split():
                    score = 80
                # Query is in tags
                elif query_lower in tags:
                    score = 70
                # Partial match in tags
                elif any(query_lower in tag for tag in tags):
                    score = 60

                if score > 0:
                    matches.append((item, score))

            # Sort by score
            matches.sort(key=lambda x: x[1], reverse=True)

            # Return items only (without scores)
            return [item for item, score in matches]

        except Exception as e:
            logger.error(f"❌ Error in simple text search: {e}")
            return []
    
    def _format_price_response(
        self,
        matching_items: List[Dict],
        item_name: str,
        lang: str
    ) -> str:
        """تنسيق الرد عن السعر"""

        if not matching_items:
            # لم يتم العثور على الصنف
            if lang == 'ar':
                return f"عذراً، لم أجد '{item_name}' في قائمتنا. هل تريد رؤية المنيو الكامل؟"
            else:
                return f"Sorry, I couldn't find '{item_name}' in our menu. Would you like to see the full menu?"

        # تم العثور على صنف واحد أو أكثر
        if lang == 'ar':
            if len(matching_items) == 1:
                # صنف واحد فقط
                item = matching_items[0]
                name = item.get('name_ar')
                # Try both 'price_omr' and 'price' keys
                price = item.get('price_omr') or item.get('price')

                if price is None:
                    return f"عذراً، لم أتمكن من العثور على سعر '{name}'. من فضلك تواصل معنا."

                reply = f"💰 **{name}** سعره **{price:.3f} ر.ع**\n\n"
                reply += "هل تريد إضافته إلى السلة؟"

            else:
                # عدة أصناف
                reply = f"لدينا عدة خيارات:\n\n"

                for i, item in enumerate(matching_items, 1):
                    name = item.get('name_ar')
                    # Try both 'price_omr' and 'price' keys
                    price = item.get('price_omr') or item.get('price')

                    if price is not None:
                        reply += f"{i}. **{name}** - {price:.3f} ر.ع\n"
                    else:
                        reply += f"{i}. **{name}** - (السعر غير متوفر)\n"

                reply += "\nأي منها تريد إضافته إلى السلة؟"

        else:
            if len(matching_items) == 1:
                # Single item
                item = matching_items[0]
                name = item.get('name_en', item.get('name_ar'))
                # Try both 'price_omr' and 'price' keys
                price = item.get('price_omr') or item.get('price')

                if price is None:
                    return f"Sorry, I couldn't find the price for '{name}'. Please contact us."

                reply = f"💰 **{name}** costs **{price:.3f} OMR**\n\n"
                reply += "Would you like to add it to your cart?"

            else:
                # Multiple items
                reply = f"We have several options:\n\n"

                for i, item in enumerate(matching_items, 1):
                    name = item.get('name_en', item.get('name_ar'))
                    # Try both 'price_omr' and 'price' keys
                    price = item.get('price_omr') or item.get('price')

                    if price is not None:
                        reply += f"{i}. **{name}** - {price:.3f} OMR\n"
                    else:
                        reply += f"{i}. **{name}** - (Price not available)\n"

                reply += "\nWhich one would you like to add to your cart?"

        return reply
    
    def _error_response(self, context: Dict) -> Dict:
        """رد عند خطأ"""
        lang = self.get_language(context)
        
        if lang == 'ar':
            reply = "عذراً، حدث خطأ في البحث عن السعر. من فضلك حاول مرة أخرى."
        else:
            reply = "Sorry, an error occurred while searching for the price. Please try again."
        
        return self.format_response(
            intent="Price Information",
            reply=reply,
            action=None,
            data=None
        )


# Singleton instance
ask_price_handler = AskPriceHandler()

