"""
Ask Menu Handler - معالج السؤال عن المنيو
"""

from typing import Dict, Optional, List
from app.handlers.base_handler import BaseIntentHandler
from app.intents.definitions import IntentType
from supabase import create_client
from app.config import settings
from app.utils.logger import app_logger as logger

# Initialize Supabase client
supabase = create_client(settings.SUPABASE_URL, settings.SUPABASE_KEY)


class AskMenuHandler(BaseIntentHandler):
    """معالج السؤال عن المنيو"""
    
    @property
    def intent_type(self) -> IntentType:
        return IntentType.ASK_MENU
    
    @property
    def prompt_template(self) -> str:
        return """أنت مساعد مطعم. مهمتك: عرض المنيو للعميل.

## القواعد:
- اعرض الأصناف مجموعة حسب الفئة
- اذكر الأسعار
- كن منظماً وواضحاً
"""
    
    async def handle(
        self,
        user_message: str,
        context: Dict,
        conversation_history: Optional[List[Dict]] = None
    ) -> Dict:
        """معالجة السؤال عن المنيو"""
        
        try:
            logger.info(f"📋 Handling ask menu")
            
            # 1. الحصول على المنيو من قاعدة البيانات
            menu_items = await self._get_menu_items()
            
            # 2. تنسيق المنيو
            lang = self.get_language(context)
            reply = self._format_menu(menu_items, lang)
            
            return self.format_response(
                intent="Menu",
                reply=reply,
                action=None,
                data=None
            )
            
        except Exception as e:
            logger.error(f"❌ Error in ask menu handler: {str(e)}")
            return self._error_response(context)
    
    async def _get_menu_items(self) -> List[Dict]:
        """الحصول على المنيو من قاعدة البيانات"""
        
        try:
            result = supabase.table("menu_items")\
                .select("*")\
                .eq("active", True)\
                .order("category", desc=False)\
                .order("sort_order", desc=False)\
                .execute()
            
            return result.data
            
        except Exception as e:
            logger.error(f"❌ Error fetching menu: {str(e)}")
            return []
    
    def _format_menu(self, menu_items: List[Dict], lang: str) -> str:
        """تنسيق المنيو"""
        
        if not menu_items:
            if lang == 'ar':
                return "عذراً، المنيو غير متاح حالياً."
            else:
                return "Sorry, the menu is not available at the moment."
        
        # تجميع حسب الفئة
        categories = {}
        for item in menu_items:
            category = item.get('category', 'أخرى')
            if category not in categories:
                categories[category] = []
            categories[category].append(item)
        
        # بناء الرد
        if lang == 'ar':
            reply = "📋 **قائمة الطعام:**\n\n"
            
            for category, items in categories.items():
                reply += f"**{category}:**\n"
                for item in items[:5]:  # أول 5 أصناف فقط من كل فئة
                    name = item.get('name_ar')
                    price = item.get('price_omr')
                    reply += f"• {name} - {price:.3f} ر.ع\n"
                reply += "\n"
            
            reply += "ماذا تريد أن تطلب؟"
            
        else:
            reply = "📋 **Menu:**\n\n"
            
            for category, items in categories.items():
                reply += f"**{category}:**\n"
                for item in items[:5]:  # First 5 items only from each category
                    name = item.get('name_en', item.get('name_ar'))
                    price = item.get('price_omr')
                    reply += f"• {name} - {price:.3f} OMR\n"
                reply += "\n"
            
            reply += "What would you like to order?"
        
        return reply
    
    def _error_response(self, context: Dict) -> Dict:
        """رد عند خطأ"""
        lang = self.get_language(context)
        
        if lang == 'ar':
            reply = "عذراً، حدث خطأ في عرض المنيو. من فضلك حاول مرة أخرى."
        else:
            reply = "Sorry, an error occurred while displaying the menu. Please try again."
        
        return self.format_response(
            intent="Menu",
            reply=reply,
            action=None,
            data=None
        )


# Singleton instance
ask_menu_handler = AskMenuHandler()

