"""
Embedding Generator
Generates embeddings for menu items using OpenAI API
"""
from typing import List, Dict, Any, Optional
import numpy as np
from openai import OpenAI
import os
from app.utils.logger import logger
from app.embeddings.text_cleaning import prepare_text_for_embedding, clean_text
import time


class EmbeddingGenerator:
    """
    Generator for creating embeddings using OpenAI API
    """
    
    def __init__(self, model: str = "text-embedding-3-small"):
        """
        Initialize embedding generator
        
        Args:
            model: OpenAI embedding model to use
        """
        self.model = model
        self.api_key = os.getenv('OPENAI_API_KEY')
        
        if not self.api_key:
            raise ValueError("OPENAI_API_KEY not found in environment")
        
        self.client = OpenAI(api_key=self.api_key)
        self.dimension = 1536  # text-embedding-3-small dimension
        
        logger.info(f"🔮 Embedding generator initialized with model: {model}")
    
    def generate_single(self, text: str) -> Optional[List[float]]:
        """
        Generate embedding for a single text
        
        Args:
            text: Input text
            
        Returns:
            Embedding vector or None if failed
        """
        if not text:
            logger.warning("⚠️  Empty text provided for embedding")
            return None
        
        try:
            # Clean text
            cleaned_text = clean_text(text)
            
            if not cleaned_text:
                logger.warning("⚠️  Text became empty after cleaning")
                return None
            
            # Generate embedding
            response = self.client.embeddings.create(
                model=self.model,
                input=cleaned_text
            )
            
            embedding = response.data[0].embedding
            
            return embedding
            
        except Exception as e:
            logger.error(f"❌ Error generating embedding: {str(e)}")
            return None
    
    def generate_batch(self, texts: List[str], batch_size: int = 100) -> List[List[float]]:
        """
        Generate embeddings for multiple texts in batches
        
        Args:
            texts: List of input texts
            batch_size: Number of texts per batch
            
        Returns:
            List of embedding vectors
        """
        if not texts:
            logger.warning("⚠️  Empty text list provided")
            return []
        
        logger.info(f"🔮 Generating embeddings for {len(texts)} texts...")
        
        all_embeddings = []
        
        # Process in batches
        for i in range(0, len(texts), batch_size):
            batch = texts[i:i + batch_size]
            batch_num = i // batch_size + 1
            total_batches = (len(texts) + batch_size - 1) // batch_size
            
            logger.info(f"📦 Processing batch {batch_num}/{total_batches} ({len(batch)} items)...")
            
            try:
                # Clean texts
                cleaned_batch = [clean_text(text) for text in batch]
                
                # Filter empty texts
                valid_texts = [text for text in cleaned_batch if text]
                
                if not valid_texts:
                    logger.warning(f"⚠️  Batch {batch_num} has no valid texts")
                    continue
                
                # Generate embeddings
                response = self.client.embeddings.create(
                    model=self.model,
                    input=valid_texts
                )
                
                # Extract embeddings
                batch_embeddings = [item.embedding for item in response.data]
                all_embeddings.extend(batch_embeddings)
                
                logger.info(f"✅ Batch {batch_num} completed ({len(batch_embeddings)} embeddings)")
                
                # Rate limiting (avoid hitting API limits)
                if i + batch_size < len(texts):
                    time.sleep(0.5)
                
            except Exception as e:
                logger.error(f"❌ Error processing batch {batch_num}: {str(e)}")
                # Continue with next batch
                continue
        
        logger.info(f"✅ Generated {len(all_embeddings)} embeddings")
        
        return all_embeddings
    
    def generate_for_menu_items(self, menu_items: List[Dict[str, Any]]) -> np.ndarray:
        """
        Generate embeddings for menu items
        
        Args:
            menu_items: List of menu item dictionaries
            
        Returns:
            Numpy array of embeddings (shape: [n_items, dimension])
        """
        logger.info(f"🔮 Generating embeddings for {len(menu_items)} menu items...")
        
        # Prepare texts
        texts = []
        for item in menu_items:
            text = prepare_text_for_embedding(item)
            texts.append(text)
        
        # Generate embeddings
        embeddings = self.generate_batch(texts)
        
        if not embeddings:
            logger.error("❌ No embeddings generated")
            return np.array([])
        
        # Convert to numpy array
        embeddings_array = np.array(embeddings, dtype='float32')
        
        logger.info(f"✅ Embeddings array shape: {embeddings_array.shape}")
        
        return embeddings_array


# Singleton instance
_embedding_generator: Optional[EmbeddingGenerator] = None


def get_embedding_generator() -> EmbeddingGenerator:
    """
    Get singleton embedding generator instance
    
    Returns:
        EmbeddingGenerator instance
    """
    global _embedding_generator
    
    if _embedding_generator is None:
        _embedding_generator = EmbeddingGenerator()
        logger.info("🔮 Embedding generator singleton created")
    
    return _embedding_generator


def generate_menu_embeddings(menu_items: List[Dict[str, Any]]) -> np.ndarray:
    """
    Generate embeddings for menu items (convenience function)
    
    Args:
        menu_items: List of menu items
        
    Returns:
        Numpy array of embeddings
    """
    generator = get_embedding_generator()
    return generator.generate_for_menu_items(menu_items)


def generate_single_embedding(text: str) -> Optional[List[float]]:
    """
    Generate embedding for single text (convenience function)
    
    Args:
        text: Input text
        
    Returns:
        Embedding vector or None
    """
    generator = get_embedding_generator()
    return generator.generate_single(text)

