"""
Unified Supabase Client
Provides a centralized interface for all Supabase operations
"""
from typing import List, Dict, Any, Optional
from supabase import create_client, Client
from app.utils.logger import logger
import os
from functools import lru_cache


class SupabaseClient:
    """
    Unified Supabase client with common operations
    """
    
    def __init__(self):
        """Initialize Supabase client"""
        self.url = os.getenv('SUPABASE_URL')
        self.key = os.getenv('SUPABASE_KEY')
        self.service_key = os.getenv('SUPABASE_SERVICE_KEY', self.key)
        
        if not self.url or not self.key:
            raise ValueError("SUPABASE_URL and SUPABASE_KEY must be set in environment")
        
        self._client: Optional[Client] = None
        self._service_client: Optional[Client] = None
    
    @property
    def client(self) -> Client:
        """Get regular client (lazy initialization)"""
        if self._client is None:
            self._client = create_client(self.url, self.key)
            logger.info("✅ Supabase client initialized")
        return self._client
    
    @property
    def service_client(self) -> Client:
        """Get service role client (lazy initialization)"""
        if self._service_client is None:
            self._service_client = create_client(self.url, self.service_key)
            logger.info("✅ Supabase service client initialized")
        return self._service_client
    
    # ==================== Menu Operations ====================
    
    def get_all_menu_items(self, active_only: bool = True) -> List[Dict[str, Any]]:
        """
        Get all menu items from database
        
        Args:
            active_only: If True, only return active items
            
        Returns:
            List of menu items
        """
        try:
            query = self.client.table('menu_items').select('*')
            
            if active_only:
                query = query.eq('active', True)
            
            result = query.order('category, sort_order').execute()
            
            logger.info(f"📊 Fetched {len(result.data)} menu items from Supabase")
            return result.data
            
        except Exception as e:
            logger.error(f"❌ Error fetching menu items: {str(e)}")
            return []
    
    def get_menu_item_by_code(self, code: str) -> Optional[Dict[str, Any]]:
        """
        Get a single menu item by code
        
        Args:
            code: Item code
            
        Returns:
            Menu item dict or None
        """
        try:
            result = self.client.table('menu_items').select('*').eq('code', code).execute()
            
            if result.data:
                return result.data[0]
            return None
            
        except Exception as e:
            logger.error(f"❌ Error fetching menu item {code}: {str(e)}")
            return None
    
    def get_menu_items_by_category(self, category: str) -> List[Dict[str, Any]]:
        """
        Get menu items by category
        
        Args:
            category: Category name
            
        Returns:
            List of menu items
        """
        try:
            result = self.client.table('menu_items').select('*')\
                .eq('category', category)\
                .eq('active', True)\
                .order('sort_order')\
                .execute()
            
            return result.data
            
        except Exception as e:
            logger.error(f"❌ Error fetching category {category}: {str(e)}")
            return []
    
    def search_menu_items(self, query: str, limit: int = 10) -> List[Dict[str, Any]]:
        """
        Search menu items by name or description
        
        Args:
            query: Search query
            limit: Maximum results
            
        Returns:
            List of matching items
        """
        try:
            # Search in name_ar, name_en, description
            result = self.client.table('menu_items').select('*')\
                .or_(f"name_ar.ilike.%{query}%,name_en.ilike.%{query}%,description.ilike.%{query}%")\
                .eq('active', True)\
                .limit(limit)\
                .execute()
            
            return result.data
            
        except Exception as e:
            logger.error(f"❌ Error searching menu: {str(e)}")
            return []
    
    def update_menu_item(self, code: str, updates: Dict[str, Any]) -> bool:
        """
        Update a menu item
        
        Args:
            code: Item code
            updates: Fields to update
            
        Returns:
            True if successful
        """
        try:
            result = self.service_client.table('menu_items')\
                .update(updates)\
                .eq('code', code)\
                .execute()
            
            logger.info(f"✅ Updated menu item {code}")
            return True
            
        except Exception as e:
            logger.error(f"❌ Error updating menu item {code}: {str(e)}")
            return False
    
    # ==================== Order Operations ====================
    
    def get_order_by_id(self, order_id: str) -> Optional[Dict[str, Any]]:
        """Get order by ID"""
        try:
            result = self.client.table('orders').select('*').eq('id', order_id).execute()
            return result.data[0] if result.data else None
        except Exception as e:
            logger.error(f"❌ Error fetching order {order_id}: {str(e)}")
            return None
    
    def get_user_orders(self, phone: str, limit: int = 10) -> List[Dict[str, Any]]:
        """Get user's order history"""
        try:
            result = self.client.table('orders').select('*')\
                .eq('customer_phone', phone)\
                .order('created_at', desc=True)\
                .limit(limit)\
                .execute()
            return result.data
        except Exception as e:
            logger.error(f"❌ Error fetching orders for {phone}: {str(e)}")
            return []
    
    def create_order(self, order_data: Dict[str, Any]) -> Optional[str]:
        """Create a new order"""
        try:
            result = self.client.table('orders').insert(order_data).execute()
            order_id = result.data[0]['id'] if result.data else None
            logger.info(f"✅ Created order {order_id}")
            return order_id
        except Exception as e:
            logger.error(f"❌ Error creating order: {str(e)}")
            return None
    
    def update_order_status(self, order_id: str, status: str) -> bool:
        """Update order status"""
        try:
            self.client.table('orders')\
                .update({'status': status})\
                .eq('id', order_id)\
                .execute()
            logger.info(f"✅ Updated order {order_id} status to {status}")
            return True
        except Exception as e:
            logger.error(f"❌ Error updating order status: {str(e)}")
            return False
    
    # ==================== Session Operations ====================
    
    def get_draft_session(self, chat_id: str) -> Optional[Dict[str, Any]]:
        """Get draft session"""
        try:
            result = self.client.table('draft_sessions').select('*')\
                .eq('chat_id', chat_id)\
                .execute()
            return result.data[0] if result.data else None
        except Exception as e:
            logger.error(f"❌ Error fetching session {chat_id}: {str(e)}")
            return None
    
    def upsert_draft_session(self, session_data: Dict[str, Any]) -> bool:
        """Create or update draft session"""
        try:
            self.client.table('draft_sessions').upsert(session_data).execute()
            return True
        except Exception as e:
            logger.error(f"❌ Error upserting session: {str(e)}")
            return False
    
    def delete_draft_session(self, chat_id: str) -> bool:
        """Delete draft session"""
        try:
            self.client.table('draft_sessions').delete().eq('chat_id', chat_id).execute()
            logger.info(f"✅ Deleted session {chat_id}")
            return True
        except Exception as e:
            logger.error(f"❌ Error deleting session: {str(e)}")
            return False
    
    # ==================== Utility Operations ====================
    
    def get_table_count(self, table_name: str) -> int:
        """Get row count for a table"""
        try:
            result = self.client.table(table_name).select('*', count='exact').execute()
            return result.count or 0
        except Exception as e:
            logger.error(f"❌ Error counting {table_name}: {str(e)}")
            return 0
    
    def health_check(self) -> bool:
        """Check if Supabase connection is healthy"""
        try:
            # Try to fetch one menu item
            result = self.client.table('menu_items').select('code').limit(1).execute()
            return len(result.data) > 0
        except Exception as e:
            logger.error(f"❌ Supabase health check failed: {str(e)}")
            return False


# Singleton instance
_supabase_client: Optional[SupabaseClient] = None


@lru_cache(maxsize=1)
def get_supabase_client() -> SupabaseClient:
    """
    Get singleton Supabase client instance
    
    Returns:
        SupabaseClient instance
    """
    global _supabase_client
    
    if _supabase_client is None:
        _supabase_client = SupabaseClient()
        logger.info("🔌 Supabase client singleton created")
    
    return _supabase_client

