"""
Configuration module for the restaurant chatbot server
"""
from pydantic_settings import BaseSettings
from typing import Optional
import os
from pathlib import Path


class Settings(BaseSettings):
    """Application settings"""
    
    # Server Configuration
    SERVER_HOST: str = "0.0.0.0"
    SERVER_PORT: int = 8000
    DEBUG: bool = False
    ENVIRONMENT: str = "production"
    
    # Supabase Configuration
    SUPABASE_URL: str
    SUPABASE_KEY: str
    SUPABASE_SERVICE_KEY: Optional[str] = None

    # Menu Configuration
    USE_LOCAL_MENU: bool = True  # Use local JSON file instead of Supabase for menu
    
    # OpenAI Configuration
    OPENAI_API_KEY: str
    OPENAI_MODEL: str = "gpt-4"
    OPENAI_TEMPERATURE: float = 0.3
    OPENAI_MAX_TOKENS: int = 1000
    
    # Evolution API Configuration
    EVOLUTION_API_URL: str
    EVOLUTION_API_KEY: str
    EVOLUTION_INSTANCE_NAME: str = "restaurant-bot"
    
    # Webhook Configuration
    WEBHOOK_SECRET: Optional[str] = None
    WEBHOOK_PATH: str = "/webhook/whatsapp"
    
    # Redis Configuration (optional)
    REDIS_HOST: str = "localhost"
    REDIS_PORT: int = 6379
    REDIS_DB: int = 0
    REDIS_PASSWORD: Optional[str] = None
    REDIS_ENABLED: bool = False
    
    # Logging
    LOG_LEVEL: str = "INFO"
    LOG_FILE: str = "logs/app.log"
    
    # Rate Limiting
    RATE_LIMIT_PER_MINUTE: int = 60
    RATE_LIMIT_PER_HOUR: int = 1000
    
    # Session Configuration
    SESSION_TIMEOUT_MINUTES: int = 30
    MAX_ITEMS_PER_ORDER: int = 20
    
    # Business Configuration
    RESTAURANT_NAME: str = "مطعم الأطباق العمانية"
    BUSINESS_HOURS_START: str = "11:00"
    BUSINESS_HOURS_END: str = "23:00"
    DELIVERY_FEE: float = 0.500
    MIN_ORDER_AMOUNT: float = 2.000

    # Testing Configuration
    TEST_PHONE_NUMBER: Optional[str] = None
    RESTAURANT_PHONE_NUMBER: Optional[str] = None

    class Config:
        env_file = ".env"
        env_file_encoding = "utf-8"
        case_sensitive = True


# Create settings instance
settings = Settings()


# Create logs directory if it doesn't exist
log_dir = Path(settings.LOG_FILE).parent
log_dir.mkdir(parents=True, exist_ok=True)

