"""
Menu Cache System
Stores menu data locally to reduce database queries and improve search performance
"""
import json
import os
from typing import List, Dict, Any, Optional
from datetime import datetime, timedelta
from pathlib import Path
from app.utils.logger import logger
from functools import lru_cache
import hashlib


class MenuCache:
    """
    In-memory and file-based cache for menu items
    """
    
    def __init__(self, cache_dir: str = "data/cache"):
        """
        Initialize menu cache
        
        Args:
            cache_dir: Directory to store cache files
        """
        self.cache_dir = Path(cache_dir)
        self.cache_dir.mkdir(parents=True, exist_ok=True)
        
        self.cache_file = self.cache_dir / "menu_cache.json"
        self.metadata_file = self.cache_dir / "menu_metadata.json"
        
        # In-memory cache
        self._menu_items: List[Dict[str, Any]] = []
        self._menu_by_code: Dict[str, Dict[str, Any]] = {}
        self._menu_by_category: Dict[str, List[Dict[str, Any]]] = {}
        self._cache_loaded = False
        self._last_update: Optional[datetime] = None
        
        logger.info(f"📦 Menu cache initialized at {self.cache_dir}")
    
    def load_from_file(self) -> bool:
        """
        Load menu cache from file
        
        Returns:
            True if loaded successfully
        """
        try:
            if not self.cache_file.exists():
                logger.warning("⚠️  Cache file not found")
                return False
            
            # Load menu data
            with open(self.cache_file, 'r', encoding='utf-8') as f:
                data = json.load(f)
            
            self._menu_items = data.get('items', [])
            
            # Load metadata
            if self.metadata_file.exists():
                with open(self.metadata_file, 'r', encoding='utf-8') as f:
                    metadata = json.load(f)
                    self._last_update = datetime.fromisoformat(metadata.get('last_update'))
            
            # Build indexes
            self._build_indexes()
            
            self._cache_loaded = True
            logger.info(f"✅ Loaded {len(self._menu_items)} items from cache")
            logger.info(f"📅 Last update: {self._last_update}")
            
            return True
            
        except Exception as e:
            logger.error(f"❌ Error loading cache: {str(e)}")
            return False
    
    def save_to_file(self, menu_items: List[Dict[str, Any]]) -> bool:
        """
        Save menu items to cache file
        
        Args:
            menu_items: List of menu items to cache
            
        Returns:
            True if saved successfully
        """
        try:
            # Save menu data
            cache_data = {
                'items': menu_items,
                'count': len(menu_items),
                'timestamp': datetime.now().isoformat()
            }
            
            with open(self.cache_file, 'w', encoding='utf-8') as f:
                json.dump(cache_data, f, ensure_ascii=False, indent=2)
            
            # Save metadata
            metadata = {
                'last_update': datetime.now().isoformat(),
                'item_count': len(menu_items),
                'checksum': self._calculate_checksum(menu_items)
            }
            
            with open(self.metadata_file, 'w', encoding='utf-8') as f:
                json.dump(metadata, f, ensure_ascii=False, indent=2)
            
            # Update in-memory cache
            self._menu_items = menu_items
            self._last_update = datetime.now()
            self._build_indexes()
            self._cache_loaded = True
            
            logger.info(f"✅ Saved {len(menu_items)} items to cache")
            
            return True
            
        except Exception as e:
            logger.error(f"❌ Error saving cache: {str(e)}")
            return False
    
    def _build_indexes(self):
        """Build internal indexes for fast lookup"""
        self._menu_by_code = {item['code']: item for item in self._menu_items}
        
        self._menu_by_category = {}
        for item in self._menu_items:
            category = item.get('category', 'Unknown')
            if category not in self._menu_by_category:
                self._menu_by_category[category] = []
            self._menu_by_category[category].append(item)
        
        logger.debug(f"📑 Built indexes: {len(self._menu_by_code)} codes, {len(self._menu_by_category)} categories")
    
    def _calculate_checksum(self, items: List[Dict[str, Any]]) -> str:
        """Calculate checksum of menu items"""
        data_str = json.dumps(items, sort_keys=True)
        return hashlib.md5(data_str.encode()).hexdigest()
    
    def get_all_items(self) -> List[Dict[str, Any]]:
        """
        Get all cached menu items
        
        Returns:
            List of menu items
        """
        if not self._cache_loaded:
            self.load_from_file()
        
        return self._menu_items.copy()
    
    def get_item_by_code(self, code: str) -> Optional[Dict[str, Any]]:
        """
        Get menu item by code
        
        Args:
            code: Item code
            
        Returns:
            Menu item or None
        """
        if not self._cache_loaded:
            self.load_from_file()
        
        return self._menu_by_code.get(code)
    
    def get_items_by_category(self, category: str) -> List[Dict[str, Any]]:
        """
        Get menu items by category
        
        Args:
            category: Category name
            
        Returns:
            List of items in category
        """
        if not self._cache_loaded:
            self.load_from_file()
        
        return self._menu_by_category.get(category, []).copy()
    
    def get_categories(self) -> List[str]:
        """
        Get all categories
        
        Returns:
            List of category names
        """
        if not self._cache_loaded:
            self.load_from_file()
        
        return list(self._menu_by_category.keys())
    
    def search_items(self, query: str, fields: List[str] = None) -> List[Dict[str, Any]]:
        """
        Search menu items
        
        Args:
            query: Search query
            fields: Fields to search in (default: name_ar, name_en, description, tags)
            
        Returns:
            List of matching items
        """
        if not self._cache_loaded:
            self.load_from_file()
        
        if fields is None:
            fields = ['name_ar', 'name_en', 'description', 'tags']
        
        query_lower = query.lower()
        results = []
        
        for item in self._menu_items:
            for field in fields:
                value = item.get(field)
                
                if value is None:
                    continue
                
                # Handle list fields (like tags)
                if isinstance(value, list):
                    if any(query_lower in str(v).lower() for v in value):
                        results.append(item)
                        break
                # Handle string fields
                elif query_lower in str(value).lower():
                    results.append(item)
                    break
        
        return results
    
    def is_stale(self, max_age_hours: int = 24) -> bool:
        """
        Check if cache is stale
        
        Args:
            max_age_hours: Maximum age in hours
            
        Returns:
            True if cache is older than max_age_hours
        """
        if not self._cache_loaded or self._last_update is None:
            return True
        
        age = datetime.now() - self._last_update
        return age > timedelta(hours=max_age_hours)
    
    def clear(self):
        """Clear cache"""
        self._menu_items = []
        self._menu_by_code = {}
        self._menu_by_category = {}
        self._cache_loaded = False
        self._last_update = None
        
        logger.info("🗑️  Cache cleared")
    
    def get_stats(self) -> Dict[str, Any]:
        """
        Get cache statistics
        
        Returns:
            Dictionary with cache stats
        """
        if not self._cache_loaded:
            self.load_from_file()
        
        return {
            'loaded': self._cache_loaded,
            'item_count': len(self._menu_items),
            'category_count': len(self._menu_by_category),
            'last_update': self._last_update.isoformat() if self._last_update else None,
            'is_stale': self.is_stale(),
            'cache_file_exists': self.cache_file.exists(),
            'cache_file_size': self.cache_file.stat().st_size if self.cache_file.exists() else 0
        }


# Singleton instance
_menu_cache: Optional[MenuCache] = None


@lru_cache(maxsize=1)
def get_menu_cache() -> MenuCache:
    """
    Get singleton menu cache instance
    
    Returns:
        MenuCache instance
    """
    global _menu_cache
    
    if _menu_cache is None:
        _menu_cache = MenuCache()
        logger.info("📦 Menu cache singleton created")
    
    return _menu_cache

