# 🚀 دليل النشر (Deployment Guide)

## خيارات النشر

### 1️⃣ VPS (Ubuntu/Debian)
### 2️⃣ Docker
### 3️⃣ Cloud Platforms (AWS, GCP, Azure)
### 4️⃣ Heroku
### 5️⃣ Railway.app

---

## 1️⃣ النشر على VPS (Ubuntu 22.04)

### الخطوة 1: تحضير السيرفر

```bash
# تحديث النظام
sudo apt update && sudo apt upgrade -y

# تثبيت Python 3.11
sudo apt install python3.11 python3.11-venv python3-pip -y

# تثبيت Git
sudo apt install git -y

# تثبيت Nginx (اختياري - للـ reverse proxy)
sudo apt install nginx -y
```

---

### الخطوة 2: رفع المشروع

```bash
# إنشاء مستخدم للتطبيق
sudo useradd -m -s /bin/bash chatbot
sudo su - chatbot

# Clone المشروع
cd ~
git clone https://github.com/your-repo/restaurant-chatbot-server.git
cd restaurant-chatbot-server

# إنشاء البيئة الافتراضية
python3.11 -m venv venv
source venv/bin/activate

# تثبيت المكتبات
pip install -r requirements.txt
```

---

### الخطوة 3: إعداد .env

```bash
nano .env
```

املأ البيانات:

```env
SERVER_HOST=0.0.0.0
SERVER_PORT=8000
DEBUG=False
ENVIRONMENT=production

SUPABASE_URL=https://yoeyovuanxrsdfgokfgh.supabase.co
SUPABASE_KEY=your_key_here

OPENAI_API_KEY=sk-proj-...

EVOLUTION_API_URL=http://your-evolution-api:8080
EVOLUTION_API_KEY=your_key_here
EVOLUTION_INSTANCE_NAME=restaurant-bot

LOG_LEVEL=INFO
```

---

### الخطوة 4: إعداد Systemd Service

```bash
sudo nano /etc/systemd/system/chatbot.service
```

```ini
[Unit]
Description=Restaurant Chatbot Server
After=network.target

[Service]
Type=simple
User=chatbot
WorkingDirectory=/home/chatbot/restaurant-chatbot-server
Environment="PATH=/home/chatbot/restaurant-chatbot-server/venv/bin"
ExecStart=/home/chatbot/restaurant-chatbot-server/venv/bin/python -m uvicorn app.main:app --host 0.0.0.0 --port 8000 --workers 4
Restart=always
RestartSec=10

[Install]
WantedBy=multi-user.target
```

```bash
# تفعيل وتشغيل الخدمة
sudo systemctl daemon-reload
sudo systemctl enable chatbot
sudo systemctl start chatbot
sudo systemctl status chatbot
```

---

### الخطوة 5: إعداد Nginx (Reverse Proxy)

```bash
sudo nano /etc/nginx/sites-available/chatbot
```

```nginx
server {
    listen 80;
    server_name your-domain.com;

    location / {
        proxy_pass http://127.0.0.1:8000;
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
}
```

```bash
# تفعيل الموقع
sudo ln -s /etc/nginx/sites-available/chatbot /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl restart nginx
```

---

### الخطوة 6: إعداد SSL (Let's Encrypt)

```bash
# تثبيت Certbot
sudo apt install certbot python3-certbot-nginx -y

# الحصول على شهادة SSL
sudo certbot --nginx -d your-domain.com

# تجديد تلقائي
sudo certbot renew --dry-run
```

---

## 2️⃣ النشر باستخدام Docker

### Dockerfile

```dockerfile
FROM python:3.11-slim

WORKDIR /app

# Install dependencies
COPY requirements.txt .
RUN pip install --no-cache-dir -r requirements.txt

# Copy application
COPY app/ ./app/

# Create logs directory
RUN mkdir -p logs

# Expose port
EXPOSE 8000

# Run application
CMD ["python", "-m", "uvicorn", "app.main:app", "--host", "0.0.0.0", "--port", "8000", "--workers", "4"]
```

### docker-compose.yml

```yaml
version: '3.8'

services:
  chatbot:
    build: .
    ports:
      - "8000:8000"
    env_file:
      - .env
    volumes:
      - ./logs:/app/logs
    restart: unless-stopped
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:8000/health"]
      interval: 30s
      timeout: 10s
      retries: 3
```

### تشغيل:

```bash
# بناء الصورة
docker-compose build

# تشغيل
docker-compose up -d

# مشاهدة Logs
docker-compose logs -f

# إيقاف
docker-compose down
```

---

## 3️⃣ النشر على Railway.app

### الخطوة 1: إنشاء حساب

1. اذهب إلى [railway.app](https://railway.app)
2. سجل دخول بـ GitHub

### الخطوة 2: إنشاء مشروع جديد

1. New Project → Deploy from GitHub repo
2. اختر repository
3. Railway سيكتشف Python تلقائياً

### الخطوة 3: إضافة Environment Variables

في Railway Dashboard:
- Variables → Add Variables
- أضف جميع المتغيرات من `.env`

### الخطوة 4: إضافة Procfile

```bash
# في جذر المشروع
echo "web: python -m uvicorn app.main:app --host 0.0.0.0 --port \$PORT" > Procfile
```

### الخطوة 5: Deploy

Railway سينشر تلقائياً!

---

## 4️⃣ النشر على Heroku

### الخطوة 1: تثبيت Heroku CLI

```bash
# Mac
brew tap heroku/brew && brew install heroku

# Ubuntu
curl https://cli-assets.heroku.com/install.sh | sh
```

### الخطوة 2: تسجيل الدخول

```bash
heroku login
```

### الخطوة 3: إنشاء تطبيق

```bash
heroku create restaurant-chatbot-app
```

### الخطوة 4: إضافة Procfile

```bash
echo "web: python -m uvicorn app.main:app --host 0.0.0.0 --port \$PORT" > Procfile
```

### الخطوة 5: إضافة runtime.txt

```bash
echo "python-3.11.7" > runtime.txt
```

### الخطوة 6: إضافة Environment Variables

```bash
heroku config:set SUPABASE_URL=https://...
heroku config:set SUPABASE_KEY=...
heroku config:set OPENAI_API_KEY=sk-...
# ... إلخ
```

### الخطوة 7: Deploy

```bash
git add .
git commit -m "Deploy to Heroku"
git push heroku main
```

---

## 🔒 أمان Production

### 1. استخدم HTTPS فقط

```python
# في main.py
from fastapi.middleware.httpsredirect import HTTPSRedirectMiddleware

if settings.ENVIRONMENT == "production":
    app.add_middleware(HTTPSRedirectMiddleware)
```

### 2. أضف Webhook Secret

```python
# في webhook.py
@router.post("/webhook/whatsapp")
async def whatsapp_webhook(request: Request):
    # Verify webhook secret
    secret = request.headers.get("X-Webhook-Secret")
    if secret != settings.WEBHOOK_SECRET:
        raise HTTPException(status_code=401, detail="Unauthorized")
    # ...
```

### 3. حدد CORS

```python
# في main.py
app.add_middleware(
    CORSMiddleware,
    allow_origins=["https://yourdomain.com"],  # حدد النطاقات المسموحة
    allow_credentials=True,
    allow_methods=["POST", "GET"],
    allow_headers=["*"],
)
```

### 4. استخدم Rate Limiting

```python
from slowapi import Limiter, _rate_limit_exceeded_handler
from slowapi.util import get_remote_address

limiter = Limiter(key_func=get_remote_address)
app.state.limiter = limiter

@router.post("/webhook/whatsapp")
@limiter.limit("60/minute")
async def whatsapp_webhook(request: Request):
    # ...
```

---

## 📊 المراقبة

### 1. Logs

```bash
# على VPS
sudo journalctl -u chatbot -f

# Docker
docker-compose logs -f

# ملف Log
tail -f logs/app.log
```

### 2. Metrics

استخدم أدوات مثل:
- **Prometheus + Grafana**
- **Datadog**
- **New Relic**

### 3. Uptime Monitoring

- **UptimeRobot**
- **Pingdom**
- **StatusCake**

---

## 🔄 التحديثات

### على VPS:

```bash
sudo su - chatbot
cd ~/restaurant-chatbot-server
git pull
source venv/bin/activate
pip install -r requirements.txt
sudo systemctl restart chatbot
```

### Docker:

```bash
docker-compose down
git pull
docker-compose build
docker-compose up -d
```

---

## 🆘 استكشاف الأخطاء

### السيرفر لا يعمل:

```bash
# تحقق من الحالة
sudo systemctl status chatbot

# شاهد Logs
sudo journalctl -u chatbot -n 100

# أعد التشغيل
sudo systemctl restart chatbot
```

### مشاكل الاتصال:

```bash
# تحقق من Firewall
sudo ufw status
sudo ufw allow 8000/tcp

# تحقق من Port
sudo netstat -tulpn | grep 8000
```

---

## ✅ Checklist للإنتاج

- [ ] DEBUG=False في .env
- [ ] HTTPS مفعّل
- [ ] Webhook Secret مضبوط
- [ ] CORS محدد
- [ ] Rate Limiting مفعّل
- [ ] Logs تُحفظ بشكل صحيح
- [ ] Monitoring مفعّل
- [ ] Backups تلقائية لقاعدة البيانات
- [ ] SSL Certificate صالح
- [ ] Firewall مضبوط
- [ ] Environment Variables آمنة

---

**جاهز للإنتاج!** 🚀

